'use client'

import Link from 'next/link'
import { useState, useEffect } from 'react'
import { isAuthenticated } from '@/lib/auth'

export default function Header() {
  const [mobileMenuOpen, setMobileMenuOpen] = useState(false)
  const [scrolled, setScrolled] = useState(false)
  const [toolsOpen, setToolsOpen] = useState(false)
  const [whatWeDoOpen, setWhatWeDoOpen] = useState(false)
  const [loggedIn, setLoggedIn] = useState(false)

  useEffect(() => {
    // Check authentication status on mount and when storage changes
    const checkAuth = () => {
      setLoggedIn(isAuthenticated())
    }
    checkAuth()
    
    // Listen for storage changes (login/logout in other tabs)
    window.addEventListener('storage', checkAuth)
    
    // Also check periodically in case of same-tab changes
    const interval = setInterval(checkAuth, 1000)
    
    return () => {
      window.removeEventListener('storage', checkAuth)
      clearInterval(interval)
    }
  }, [])

  useEffect(() => {
    const handleScroll = () => {
      setScrolled(window.scrollY > 20)
    }
    window.addEventListener('scroll', handleScroll)
    return () => window.removeEventListener('scroll', handleScroll)
  }, [])

  useEffect(() => {
    const handleResize = () => {
      if (window.innerWidth >= 1024) setMobileMenuOpen(false)
    }
    window.addEventListener('resize', handleResize)
    return () => window.removeEventListener('resize', handleResize)
  }, [])

  // Prevent body scroll when mobile menu is open
  useEffect(() => {
    if (mobileMenuOpen) {
      document.body.style.overflow = 'hidden'
    } else {
      document.body.style.overflow = 'unset'
    }
    return () => {
      document.body.style.overflow = 'unset'
    }
  }, [mobileMenuOpen])

  const navLinks = [
    { href: '/', label: 'Home' },
    { href: '/about', label: 'About' },
    { href: '/members', label: 'Members' },
    { href: '/events', label: 'Events' },
    { href: '/resources', label: 'Resources' },
  ]

  const toolsLinks = [
    { href: '/tools/3w-mapping', label: '3W Mapping', icon: '🗺️', description: 'Interactive presence mapping' },
    { href: '/jobs', label: 'Jobs & Opportunities', icon: '💼', description: 'Career openings & tenders' },
    { href: '/forum', label: 'Discussion Forum', icon: '💬', description: 'Community discussions' },
  ]

  const whatWeDoLinks = [
    { href: '/what-we-do/information-sharing', label: 'Information Sharing', icon: '📢', description: 'Disseminating relevant information' },
    { href: '/what-we-do/security', label: 'Safety & Security', icon: '🛡️', description: 'Security analysis and guidance' },
    { href: '/what-we-do/policy', label: 'Policy', icon: '📋', description: 'Policy engagement and advocacy' },
    { href: '/what-we-do/external-engagement', label: 'External Engagement', icon: '🤝', description: 'Stakeholder partnerships' },
    { href: '/what-we-do/ingo-nngo-networking', label: 'INGO-NNGO Networking', icon: '🔗', description: 'Capacity development' },
  ]

  return (
    <>
      <header 
        className={`fixed top-0 left-0 right-0 z-50 transition-all duration-300 ${
          scrolled 
            ? 'bg-white/95 backdrop-blur-xl shadow-lg shadow-secondary/5 border-b border-primary/10' 
            : 'bg-white/80 backdrop-blur-md border-b border-transparent'
        }`}
      >
        <nav className="container mx-auto px-4 lg:px-8">
          <div className="flex justify-between items-center h-16 lg:h-20">
            {/* Logo */}
            <Link href="/" className="flex items-center space-x-3 group">
              <div className="relative">
                <div className="w-10 h-10 lg:w-12 lg:h-12 rounded-xl bg-gradient-to-br from-primary via-primary-light to-secondary flex items-center justify-center shadow-lg shadow-primary/20 group-hover:shadow-xl group-hover:shadow-primary/30 transition-all duration-300 group-hover:scale-105">
                  <span className="text-white font-bold text-lg lg:text-xl">SS</span>
                </div>
                <div className="absolute inset-0 rounded-xl bg-gradient-to-br from-primary to-secondary opacity-0 group-hover:opacity-20 blur-xl transition-opacity duration-300" />
              </div>
              <div className="hidden sm:block">
                <div className="text-lg lg:text-xl font-bold text-secondary group-hover:text-primary transition-colors duration-300">
                  South Sudan
                </div>
                <div className="text-xs lg:text-sm text-text-secondary font-medium -mt-0.5">NGO Forum</div>
              </div>
            </Link>

            {/* Desktop Navigation */}
            <div className="hidden lg:flex items-center space-x-1">
              {navLinks.map((link) => (
                <Link 
                  key={link.href}
                  href={link.href} 
                  className="relative px-4 py-2 text-sm font-medium text-text-secondary hover:text-primary transition-colors duration-200 group"
                >
                  {link.label}
                  <span className="absolute bottom-0 left-1/2 -translate-x-1/2 w-0 h-0.5 bg-gradient-to-r from-primary to-primary-light group-hover:w-3/4 transition-all duration-300 rounded-full" />
                </Link>
              ))}

              {/* What We Do Dropdown */}
              <div 
                className="relative"
                onMouseEnter={() => setWhatWeDoOpen(true)}
                onMouseLeave={() => setWhatWeDoOpen(false)}
              >
                <button className="flex items-center px-4 py-2 text-sm font-medium text-text-secondary hover:text-primary transition-colors duration-200 group">
                  What We Do
                  <svg className={`ml-1 w-4 h-4 transition-transform duration-200 ${whatWeDoOpen ? 'rotate-180' : ''}`} fill="none" viewBox="0 0 24 24" stroke="currentColor">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 9l-7 7-7-7" />
                  </svg>
                </button>

                <div className={`absolute top-full left-0 mt-2 w-80 transition-all duration-300 ${whatWeDoOpen ? 'opacity-100 visible translate-y-0' : 'opacity-0 invisible -translate-y-2'}`}>
                  <div className="absolute -top-2 left-0 right-0 h-4" />
                  <div className="bg-white rounded-2xl p-2 shadow-2xl shadow-secondary/10 border border-primary/10">
                    {whatWeDoLinks.map((item, index) => (
                      <Link 
                        key={item.href} 
                        href={item.href} 
                        className={`flex items-start gap-3 px-4 py-3 rounded-xl text-secondary hover:bg-gradient-to-r hover:from-primary/5 hover:to-transparent transition-all duration-200 group ${index > 0 ? 'mt-1' : ''}`}
                      >
                        <span className="text-2xl group-hover:scale-110 transition-transform duration-200 mt-0.5">{item.icon}</span>
                        <div>
                          <span className="font-semibold text-secondary group-hover:text-primary transition-colors">{item.label}</span>
                          <p className="text-xs text-text-muted mt-0.5">{item.description}</p>
                        </div>
                      </Link>
                    ))}
                  </div>
                </div>
              </div>

              {/* Tools Dropdown */}
              <div 
                className="relative"
                onMouseEnter={() => setToolsOpen(true)}
                onMouseLeave={() => setToolsOpen(false)}
              >
                <button className="flex items-center px-4 py-2 text-sm font-medium text-text-secondary hover:text-primary transition-colors duration-200 group">
                  Tools
                  <svg className={`ml-1 w-4 h-4 transition-transform duration-200 ${toolsOpen ? 'rotate-180' : ''}`} fill="none" viewBox="0 0 24 24" stroke="currentColor">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 9l-7 7-7-7" />
                  </svg>
                </button>

                <div className={`absolute top-full left-0 mt-2 w-72 transition-all duration-300 ${toolsOpen ? 'opacity-100 visible translate-y-0' : 'opacity-0 invisible -translate-y-2'}`}>
                  <div className="absolute -top-2 left-0 right-0 h-4" />
                  <div className="bg-white rounded-2xl p-2 shadow-2xl shadow-secondary/10 border border-primary/10">
                    {toolsLinks.map((tool, index) => (
                      <Link 
                        key={tool.href} 
                        href={tool.href} 
                        className={`flex items-start gap-3 px-4 py-3 rounded-xl text-secondary hover:bg-gradient-to-r hover:from-primary/5 hover:to-transparent transition-all duration-200 group ${index > 0 ? 'mt-1' : ''}`}
                      >
                        <span className="text-2xl group-hover:scale-110 transition-transform duration-200 mt-0.5">{tool.icon}</span>
                        <div>
                          <span className="font-semibold text-secondary group-hover:text-primary transition-colors">{tool.label}</span>
                          <p className="text-xs text-text-muted mt-0.5">{tool.description}</p>
                        </div>
                      </Link>
                    ))}
                  </div>
                </div>
              </div>

              <Link href="/contact" className="px-4 py-2 text-sm font-medium text-text-secondary hover:text-primary transition-colors duration-200">
                Contact
              </Link>
            </div>

            {/* Desktop CTA */}
            <div className="hidden lg:flex items-center space-x-3">
              {!loggedIn && (
                <Link href="/apply" className="text-sm font-semibold text-primary hover:text-primary-dark transition-colors duration-200 px-4 py-2">
                  Join Us
                </Link>
              )}
              <Link href={loggedIn ? "/portal/dashboard" : "/portal/login"} className="relative group px-6 py-2.5 rounded-xl bg-gradient-to-r from-primary to-primary-light text-white font-semibold text-sm overflow-hidden transition-all duration-300 hover:shadow-lg hover:shadow-primary/30 hover:scale-[1.02] active:scale-[0.98]">
                <span className="relative z-10">{loggedIn ? "Dashboard" : "Member Login"}</span>
                <div className="absolute inset-0 bg-gradient-to-r from-primary-light to-primary opacity-0 group-hover:opacity-100 transition-opacity duration-300" />
              </Link>
            </div>

            {/* Mobile Menu Button */}
            <button 
              className="lg:hidden relative w-12 h-12 flex items-center justify-center text-secondary rounded-xl hover:bg-primary/5 active:bg-primary/10 transition-colors duration-200"
              onClick={() => setMobileMenuOpen(!mobileMenuOpen)}
              aria-label="Toggle menu"
            >
              <div className="w-6 h-5 relative flex flex-col justify-between">
                <span className={`w-full h-0.5 bg-current rounded-full transform transition-all duration-300 origin-center ${mobileMenuOpen ? 'rotate-45 translate-y-2' : ''}`} />
                <span className={`w-full h-0.5 bg-current rounded-full transition-all duration-300 ${mobileMenuOpen ? 'opacity-0 scale-0' : ''}`} />
                <span className={`w-full h-0.5 bg-current rounded-full transform transition-all duration-300 origin-center ${mobileMenuOpen ? '-rotate-45 -translate-y-2' : ''}`} />
              </div>
            </button>
          </div>
        </nav>
      </header>

      {/* Mobile Menu Overlay */}
      <div className={`fixed inset-0 z-40 lg:hidden transition-all duration-500 ${mobileMenuOpen ? 'visible' : 'invisible'}`}>
        <div className={`absolute inset-0 bg-secondary/60 backdrop-blur-sm transition-opacity duration-500 ${mobileMenuOpen ? 'opacity-100' : 'opacity-0'}`} onClick={() => setMobileMenuOpen(false)} />
        <div className={`absolute top-0 right-0 w-full max-w-sm h-full bg-white shadow-2xl transform transition-transform duration-500 ease-out ${mobileMenuOpen ? 'translate-x-0' : 'translate-x-full'}`}>
          <div className="flex justify-end p-4">
            <button onClick={() => setMobileMenuOpen(false)} className="w-10 h-10 flex items-center justify-center rounded-full hover:bg-gray-100 transition-colors">
              <svg className="w-6 h-6 text-secondary" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M6 18L18 6M6 6l12 12" />
              </svg>
            </button>
          </div>
          <div className="flex flex-col h-[calc(100%-60px)] pb-8 px-6 overflow-y-auto">
            <nav className="flex-1 space-y-1">
              {navLinks.map((link) => (
                <Link key={link.href} href={link.href} onClick={() => setMobileMenuOpen(false)} className="flex items-center py-4 text-lg font-medium text-secondary hover:text-primary border-b border-gray-100 transition-all duration-200">
                  {link.label}
                  <svg className="w-5 h-5 ml-auto text-gray-400" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 5l7 7-7 7" />
                  </svg>
                </Link>
              ))}
              <div className="pt-6 pb-2">
                <div className="text-xs font-bold text-primary uppercase tracking-wider mb-4">What We Do</div>
                <div className="space-y-2">
                  {whatWeDoLinks.map((item) => (
                    <Link key={item.href} href={item.href} onClick={() => setMobileMenuOpen(false)} className="flex items-center gap-4 py-3 px-4 rounded-xl text-secondary hover:bg-primary/5 active:bg-primary/10 transition-colors duration-200">
                      <span className="text-2xl">{item.icon}</span>
                      <div>
                        <span className="font-medium">{item.label}</span>
                        <p className="text-xs text-text-muted">{item.description}</p>
                      </div>
                    </Link>
                  ))}
                </div>
              </div>
              <div className="pt-6 pb-2">
                <div className="text-xs font-bold text-primary uppercase tracking-wider mb-4">Tools & Features</div>
                <div className="space-y-2">
                  {toolsLinks.map((tool) => (
                    <Link key={tool.href} href={tool.href} onClick={() => setMobileMenuOpen(false)} className="flex items-center gap-4 py-3 px-4 rounded-xl text-secondary hover:bg-primary/5 active:bg-primary/10 transition-colors duration-200">
                      <span className="text-2xl">{tool.icon}</span>
                      <div>
                        <span className="font-medium">{tool.label}</span>
                        <p className="text-xs text-text-muted">{tool.description}</p>
                      </div>
                    </Link>
                  ))}
                </div>
              </div>
              <Link href="/contact" onClick={() => setMobileMenuOpen(false)} className="flex items-center py-4 text-lg font-medium text-secondary hover:text-primary border-b border-gray-100 transition-colors duration-200">
                Contact
                <svg className="w-5 h-5 ml-auto text-gray-400" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 5l7 7-7 7" />
                </svg>
              </Link>
            </nav>
            <div className="space-y-3 pt-6 border-t border-gray-100 mt-4">
              {!loggedIn && (
                <Link href="/apply" onClick={() => setMobileMenuOpen(false)} className="block w-full py-4 text-center text-primary font-bold border-2 border-primary rounded-2xl hover:bg-primary/5 active:bg-primary/10 transition-colors duration-200">
                  Become a Member
                </Link>
              )}
              <Link href={loggedIn ? "/portal/dashboard" : "/portal/login"} onClick={() => setMobileMenuOpen(false)} className="block w-full py-4 text-center bg-gradient-to-r from-primary to-primary-light text-white font-bold rounded-2xl shadow-lg shadow-primary/20 active:scale-[0.98] transition-transform duration-200">
                {loggedIn ? "Dashboard" : "Member Login"}
              </Link>
            </div>
          </div>
        </div>
      </div>

      <div className="h-16 lg:h-20" />
    </>
  )
}
