'use client'

import { useState, useEffect, useRef } from 'react'

interface FAQ {
  id: number
  question: string
  answer: string
  category: string
}

const faqs: FAQ[] = [
  {
    id: 1,
    question: "How do I become a member?",
    answer: "To become a member, click on 'Join Us' or 'Become a Member' button and fill out the membership application form. Our team will review your application and get back to you within 5-7 business days.",
    category: "Membership"
  },
  {
    id: 2,
    question: "What is the 3W Mapping tool?",
    answer: "The 3W (Who does What Where) Mapping tool is an interactive visualization that shows humanitarian activities across South Sudan. It helps organizations coordinate their efforts and avoid duplication of services.",
    category: "Tools"
  },
  {
    id: 3,
    question: "How can I access member resources?",
    answer: "Member resources are available through the Member Portal. Log in with your credentials to access documents, reports, and exclusive content. If you've forgotten your password, use the 'Forgot Password' link.",
    category: "Resources"
  },
  {
    id: 4,
    question: "How do I post a job or tender?",
    answer: "Members can post jobs, training opportunities, and tenders through the Member Portal. Navigate to the Jobs section and click 'Post New Opportunity'. All postings are reviewed before publishing.",
    category: "Jobs"
  },
  {
    id: 5,
    question: "How can I attend coordination meetings?",
    answer: "Coordination meetings are listed in the Events section. Members receive email notifications about upcoming meetings. You can also subscribe to our calendar for automatic updates.",
    category: "Events"
  },
  {
    id: 6,
    question: "Who can I contact for support?",
    answer: "For general inquiries, email info@southsudanngoforum.org. For technical support, email support@southsudanngoforum.org. You can also visit our Contact page for more options.",
    category: "Support"
  },
  {
    id: 7,
    question: "What are the membership benefits?",
    answer: "Members get access to: coordination meetings, 3W data submission, resource library, job posting, forum discussions, security updates, and networking opportunities with 200+ organizations.",
    category: "Membership"
  },
  {
    id: 8,
    question: "How do I update my organization's profile?",
    answer: "Log into the Member Portal and navigate to 'Profile'. You can update your organization's information, contact details, and operational areas. Changes are reflected after admin approval.",
    category: "Account"
  }
]

interface Message {
  id: number
  type: 'bot' | 'user' | 'faq'
  content: string
  timestamp: Date
  faqId?: number
}

export default function ChatWidget() {
  const [isOpen, setIsOpen] = useState(false)
  const [messages, setMessages] = useState<Message[]>([])
  const [showFAQs, setShowFAQs] = useState(true)
  const [selectedCategory, setSelectedCategory] = useState<string | null>(null)
  const [inputValue, setInputValue] = useState('')
  const [isTyping, setIsTyping] = useState(false)
  const messagesEndRef = useRef<HTMLDivElement>(null)

  const categories = Array.from(new Set(faqs.map(f => f.category)))

  useEffect(() => {
    if (isOpen && messages.length === 0) {
      // Initial greeting
      setMessages([{
        id: 1,
        type: 'bot',
        content: "👋 Hello! I'm here to help you navigate the South Sudan NGO Forum. Choose a topic below or ask me a question!",
        timestamp: new Date()
      }])
    }
  }, [isOpen, messages.length])

  useEffect(() => {
    messagesEndRef.current?.scrollIntoView({ behavior: 'smooth' })
  }, [messages])

  const handleFAQClick = (faq: FAQ) => {
    setIsTyping(true)
    
    // Add user's question
    const userMessage: Message = {
      id: messages.length + 1,
      type: 'user',
      content: faq.question,
      timestamp: new Date()
    }
    setMessages(prev => [...prev, userMessage])
    setShowFAQs(false)

    // Simulate typing delay
    setTimeout(() => {
      setIsTyping(false)
      const botMessage: Message = {
        id: messages.length + 2,
        type: 'bot',
        content: faq.answer,
        timestamp: new Date(),
        faqId: faq.id
      }
      setMessages(prev => [...prev, botMessage])
      
      // Show FAQs again after a delay
      setTimeout(() => setShowFAQs(true), 500)
    }, 800)
  }

  const handleSendMessage = () => {
    if (!inputValue.trim()) return

    const userMessage: Message = {
      id: messages.length + 1,
      type: 'user',
      content: inputValue,
      timestamp: new Date()
    }
    setMessages(prev => [...prev, userMessage])
    setInputValue('')
    setIsTyping(true)
    setShowFAQs(false)

    // Simple keyword matching for demo
    setTimeout(() => {
      setIsTyping(false)
      const lowerInput = inputValue.toLowerCase()
      let response = "I'm still learning! For now, please check our FAQs below or contact us at info@southsudanngoforum.org for assistance."
      
      if (lowerInput.includes('member') || lowerInput.includes('join')) {
        response = faqs.find(f => f.id === 1)?.answer || response
      } else if (lowerInput.includes('3w') || lowerInput.includes('map')) {
        response = faqs.find(f => f.id === 2)?.answer || response
      } else if (lowerInput.includes('job') || lowerInput.includes('tender')) {
        response = faqs.find(f => f.id === 4)?.answer || response
      } else if (lowerInput.includes('event') || lowerInput.includes('meeting')) {
        response = faqs.find(f => f.id === 5)?.answer || response
      } else if (lowerInput.includes('contact') || lowerInput.includes('support') || lowerInput.includes('help')) {
        response = faqs.find(f => f.id === 6)?.answer || response
      } else if (lowerInput.includes('resource') || lowerInput.includes('document')) {
        response = faqs.find(f => f.id === 3)?.answer || response
      } else if (lowerInput.includes('benefit')) {
        response = faqs.find(f => f.id === 7)?.answer || response
      } else if (lowerInput.includes('profile') || lowerInput.includes('update')) {
        response = faqs.find(f => f.id === 8)?.answer || response
      } else if (lowerInput.includes('hello') || lowerInput.includes('hi') || lowerInput.includes('hey')) {
        response = "Hello! 👋 How can I help you today? Feel free to ask about membership, events, resources, or anything else!"
      } else if (lowerInput.includes('thank')) {
        response = "You're welcome! 😊 Is there anything else I can help you with?"
      }

      const botMessage: Message = {
        id: messages.length + 2,
        type: 'bot',
        content: response,
        timestamp: new Date()
      }
      setMessages(prev => [...prev, botMessage])
      setTimeout(() => setShowFAQs(true), 500)
    }, 1000)
  }

  const filteredFAQs = selectedCategory 
    ? faqs.filter(f => f.category === selectedCategory)
    : faqs.slice(0, 4)

  return (
    <>
      {/* Chat Button */}
      <button
        onClick={() => setIsOpen(!isOpen)}
        className={`fixed bottom-6 right-6 z-50 w-14 h-14 rounded-full shadow-lg transition-all duration-300 flex items-center justify-center ${
          isOpen 
            ? 'bg-secondary rotate-0 scale-100' 
            : 'bg-gradient-to-r from-primary to-primary-light hover:shadow-xl hover:shadow-primary/30 hover:scale-110'
        }`}
        aria-label={isOpen ? 'Close chat' : 'Open chat'}
      >
        {isOpen ? (
          <svg className="w-6 h-6 text-white" fill="none" viewBox="0 0 24 24" stroke="currentColor">
            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M6 18L18 6M6 6l12 12" />
          </svg>
        ) : (
          <svg className="w-6 h-6 text-white" fill="none" viewBox="0 0 24 24" stroke="currentColor">
            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M8 12h.01M12 12h.01M16 12h.01M21 12c0 4.418-4.03 8-9 8a9.863 9.863 0 01-4.255-.949L3 20l1.395-3.72C3.512 15.042 3 13.574 3 12c0-4.418 4.03-8 9-8s9 3.582 9 8z" />
          </svg>
        )}
        
        {/* Notification dot */}
        {!isOpen && (
          <span className="absolute -top-1 -right-1 w-4 h-4 bg-rose-500 rounded-full flex items-center justify-center animate-pulse">
            <span className="text-[10px] text-white font-bold">!</span>
          </span>
        )}
      </button>

      {/* Chat Window */}
      <div className={`fixed bottom-24 right-6 z-50 w-[380px] max-w-[calc(100vw-48px)] transition-all duration-300 ${
        isOpen ? 'opacity-100 translate-y-0 pointer-events-auto' : 'opacity-0 translate-y-4 pointer-events-none'
      }`}>
        <div className="bg-white rounded-2xl shadow-2xl border border-border overflow-hidden flex flex-col" style={{ height: '520px' }}>
          {/* Header */}
          <div className="bg-gradient-to-r from-primary to-primary-light p-4 flex items-center gap-3">
            <div className="w-10 h-10 rounded-full bg-white/20 flex items-center justify-center">
              <svg className="w-6 h-6 text-white" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9.663 17h4.673M12 3v1m6.364 1.636l-.707.707M21 12h-1M4 12H3m3.343-5.657l-.707-.707m2.828 9.9a5 5 0 117.072 0l-.548.547A3.374 3.374 0 0014 18.469V19a2 2 0 11-4 0v-.531c0-.895-.356-1.754-.988-2.386l-.548-.547z" />
              </svg>
            </div>
            <div className="flex-1">
              <h3 className="text-white font-bold">NGO Forum Assistant</h3>
              <p className="text-white/80 text-sm flex items-center gap-1.5">
                <span className="w-2 h-2 bg-green-400 rounded-full animate-pulse" />
                Online • Ready to help
              </p>
            </div>
            <button 
              onClick={() => setIsOpen(false)}
              className="w-8 h-8 rounded-full hover:bg-white/20 flex items-center justify-center transition-colors"
            >
              <svg className="w-5 h-5 text-white" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 9l-7 7-7-7" />
              </svg>
            </button>
          </div>

          {/* Messages */}
          <div className="flex-1 overflow-y-auto p-4 space-y-4 bg-gradient-to-b from-background to-white">
            {messages.map((message) => (
              <div
                key={message.id}
                className={`flex ${message.type === 'user' ? 'justify-end' : 'justify-start'}`}
              >
                <div className={`max-w-[85%] ${message.type === 'user' ? 'order-1' : 'order-2'}`}>
                  {message.type === 'bot' && (
                    <div className="flex items-center gap-2 mb-1">
                      <div className="w-6 h-6 rounded-full bg-primary/10 flex items-center justify-center">
                        <svg className="w-3.5 h-3.5 text-primary" fill="currentColor" viewBox="0 0 24 24">
                          <path d="M12 2C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2zm-1 17.93c-3.95-.49-7-3.85-7-7.93 0-.62.08-1.21.21-1.79L9 15v1c0 1.1.9 2 2 2v1.93zm6.9-2.54c-.26-.81-1-1.39-1.9-1.39h-1v-3c0-.55-.45-1-1-1H8v-2h2c.55 0 1-.45 1-1V7h2c1.1 0 2-.9 2-2v-.41c2.93 1.19 5 4.06 5 7.41 0 2.08-.8 3.97-2.1 5.39z"/>
                        </svg>
                      </div>
                      <span className="text-xs text-secondary/50">Assistant</span>
                    </div>
                  )}
                  <div className={`rounded-2xl px-4 py-3 ${
                    message.type === 'user'
                      ? 'bg-primary text-white rounded-br-md'
                      : 'bg-white border border-border shadow-sm rounded-bl-md'
                  }`}>
                    <p className={`text-sm leading-relaxed ${message.type === 'user' ? 'text-white' : 'text-secondary'}`}>
                      {message.content}
                    </p>
                  </div>
                  <p className={`text-[10px] mt-1 ${message.type === 'user' ? 'text-right' : 'text-left'} text-secondary/40`}>
                    {message.timestamp.toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' })}
                  </p>
                </div>
              </div>
            ))}

            {/* Typing indicator */}
            {isTyping && (
              <div className="flex justify-start">
                <div className="bg-white border border-border rounded-2xl rounded-bl-md px-4 py-3 shadow-sm">
                  <div className="flex gap-1.5">
                    <span className="w-2 h-2 bg-secondary/30 rounded-full animate-bounce" style={{ animationDelay: '0ms' }} />
                    <span className="w-2 h-2 bg-secondary/30 rounded-full animate-bounce" style={{ animationDelay: '150ms' }} />
                    <span className="w-2 h-2 bg-secondary/30 rounded-full animate-bounce" style={{ animationDelay: '300ms' }} />
                  </div>
                </div>
              </div>
            )}

            <div ref={messagesEndRef} />
          </div>

          {/* FAQ Suggestions */}
          {showFAQs && !isTyping && (
            <div className="border-t border-border bg-background/50 p-3">
              {/* Category filters */}
              <div className="flex gap-2 mb-2 overflow-x-auto hide-scrollbar pb-1">
                <button
                  onClick={() => setSelectedCategory(null)}
                  className={`px-3 py-1 rounded-full text-xs font-medium whitespace-nowrap transition-colors ${
                    selectedCategory === null
                      ? 'bg-primary text-white'
                      : 'bg-white border border-border text-secondary/70 hover:border-primary/30'
                  }`}
                >
                  Popular
                </button>
                {categories.map((cat) => (
                  <button
                    key={cat}
                    onClick={() => setSelectedCategory(cat)}
                    className={`px-3 py-1 rounded-full text-xs font-medium whitespace-nowrap transition-colors ${
                      selectedCategory === cat
                        ? 'bg-primary text-white'
                        : 'bg-white border border-border text-secondary/70 hover:border-primary/30'
                    }`}
                  >
                    {cat}
                  </button>
                ))}
              </div>
              
              {/* FAQ buttons */}
              <div className="space-y-1.5 max-h-28 overflow-y-auto">
                {filteredFAQs.map((faq) => (
                  <button
                    key={faq.id}
                    onClick={() => handleFAQClick(faq)}
                    className="w-full text-left px-3 py-2 rounded-lg bg-white border border-border hover:border-primary/30 hover:bg-primary/5 transition-all text-xs text-secondary/80 hover:text-secondary flex items-center gap-2 group"
                  >
                    <svg className="w-4 h-4 text-primary/50 group-hover:text-primary flex-shrink-0" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M8.228 9c.549-1.165 2.03-2 3.772-2 2.21 0 4 1.343 4 3 0 1.4-1.278 2.575-3.006 2.907-.542.104-.994.54-.994 1.093m0 3h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z" />
                    </svg>
                    <span className="truncate">{faq.question}</span>
                  </button>
                ))}
              </div>
            </div>
          )}

          {/* Input */}
          <div className="border-t border-border p-3 bg-white">
            <div className="flex gap-2">
              <input
                type="text"
                value={inputValue}
                onChange={(e) => setInputValue(e.target.value)}
                onKeyDown={(e) => e.key === 'Enter' && handleSendMessage()}
                placeholder="Type your question..."
                className="flex-1 px-4 py-2.5 rounded-xl border border-border bg-background focus:outline-none focus:border-primary focus:ring-2 focus:ring-primary/10 text-sm text-secondary placeholder:text-secondary/40"
              />
              <button
                onClick={handleSendMessage}
                disabled={!inputValue.trim()}
                className="w-10 h-10 rounded-xl bg-primary text-white flex items-center justify-center hover:bg-primary-dark transition-colors disabled:opacity-50 disabled:cursor-not-allowed"
              >
                <svg className="w-5 h-5" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 19l9 2-9-18-9 18 9-2zm0 0v-8" />
                </svg>
              </button>
            </div>
            <p className="text-[10px] text-secondary/40 text-center mt-2">
              Powered by South Sudan NGO Forum
            </p>
          </div>
        </div>
      </div>
    </>
  )
}
