'use client'

import { useEffect, useState } from 'react'
import { useRouter } from 'next/navigation'
import Link from 'next/link'
import { isAuthenticated, removeAuthTokens } from '@/lib/auth'
import api from '@/lib/api'

// Helper function to format dates nicely
const formatDate = (dateString: string | null): string => {
  if (!dateString) return 'N/A'
  const date = new Date(dateString)
  return date.toLocaleDateString('en-US', { year: 'numeric', month: 'long', day: 'numeric' })
}

// Check if membership is expired
const isExpired = (dateString: string | null): boolean => {
  if (!dateString) return false
  const expiryDate = new Date(dateString)
  return expiryDate < new Date()
}

interface Organization {
  id: number
  name: string
  slug: string
  member_type: string
  email: string
  phone: string
  website: string
  address: string
  city: string
  state: string
  description: string
  logo: string | null
  status: string
  is_verified: boolean
  membership_fee_paid: boolean
  membership_expiry_date: string | null
  date_joined: string
}

interface Application {
  application_id: string
  application_status: string
  submitted_date: string
  reviewed_date: string | null
  organization_name: string
  invoicing_contact_email: string
  invoicing_contact_phone: string
}

interface StatusInfo {
  status: string
  status_label: string
  status_color: string
  message: string
  can_pay: boolean
  next_steps: string[]
}

interface PendingPayment {
  external_id: string
  amount: string
  currency: string
  payer_phone: string
  requested_at: string
}

interface MemberStatus {
  success: boolean
  has_organization: boolean
  organization: Organization | null
  application: Application | null
  pending_payments: PendingPayment[]
  status_info: StatusInfo
  membership_fee: string
  currency: string
}

export default function Dashboard() {
  const router = useRouter()
  const [memberStatus, setMemberStatus] = useState<MemberStatus | null>(null)
  const [loading, setLoading] = useState(true)
  const [paymentLoading, setPaymentLoading] = useState(false)
  const [phoneNumber, setPhoneNumber] = useState('')
  const [showPaymentModal, setShowPaymentModal] = useState(false)
  const [paymentMessage, setPaymentMessage] = useState<{ type: 'success' | 'error', text: string } | null>(null)
  const [checkingPayment, setCheckingPayment] = useState(false)
  const [error, setError] = useState<string | null>(null)

  useEffect(() => {
    if (!isAuthenticated()) {
      router.push('/portal/login')
      return
    }

    fetchMemberStatus()
  }, [])

  const fetchMemberStatus = async () => {
    try {
      setError(null)
      const response = await api.get('/status/')
      console.log('Member status response:', response.data)
      setMemberStatus(response.data)
      
      // Pre-fill phone number if available
      if (response.data.application?.invoicing_contact_phone) {
        setPhoneNumber(response.data.application.invoicing_contact_phone)
      }
    } catch (err: any) {
      console.error('Failed to fetch member status:', err)
      const errorMessage = err.response?.data?.detail || err.response?.data?.error || err.message || 'Failed to load dashboard data'
      setError(errorMessage)
    } finally {
      setLoading(false)
    }
  }

  const handleLogout = () => {
    removeAuthTokens()
    router.push('/portal/login')
  }

  const initiatePayment = async () => {
    if (!phoneNumber) {
      setPaymentMessage({ type: 'error', text: 'Please enter a phone number' })
      return
    }

    setPaymentLoading(true)
    setPaymentMessage(null)

    try {
      const response = await api.post('/momo/initiate/', {
        application_id: memberStatus?.application?.application_id,
        phone_number: phoneNumber
      })

      if (response.data.success) {
        setPaymentMessage({ 
          type: 'success', 
          text: 'Payment request sent! Please check your phone and enter your PIN to approve the payment.' 
        })
        // Refresh status after a few seconds
        setTimeout(() => {
          fetchMemberStatus()
        }, 5000)
      } else {
        setPaymentMessage({ type: 'error', text: response.data.error || 'Failed to initiate payment' })
      }
    } catch (error: any) {
      setPaymentMessage({ 
        type: 'error', 
        text: error.response?.data?.error || 'Failed to initiate payment. Please try again.' 
      })
    } finally {
      setPaymentLoading(false)
    }
  }

  const checkPaymentStatus = async (externalId: string) => {
    setCheckingPayment(true)
    try {
      const response = await api.get(`/momo/status/${externalId}/`)
      if (response.data.status === 'SUCCESSFUL') {
        setPaymentMessage({ type: 'success', text: 'Payment successful! Your membership is now active.' })
        fetchMemberStatus()
      } else if (response.data.status === 'FAILED') {
        setPaymentMessage({ type: 'error', text: 'Payment failed. Please try again.' })
      } else {
        setPaymentMessage({ type: 'success', text: `Payment status: ${response.data.status}. Please wait...` })
      }
    } catch (error) {
      console.error('Failed to check payment status:', error)
    } finally {
      setCheckingPayment(false)
    }
  }

  const getStatusColorClass = (color: string) => {
    switch (color) {
      case 'green': return 'bg-green-100 text-green-800 border-green-200'
      case 'yellow': return 'bg-yellow-100 text-yellow-800 border-yellow-200'
      case 'red': return 'bg-red-100 text-red-800 border-red-200'
      default: return 'bg-gray-100 text-gray-800 border-gray-200'
    }
  }

  const getStatusBadgeClass = (color: string) => {
    switch (color) {
      case 'green': return 'bg-green-500'
      case 'yellow': return 'bg-yellow-500'
      case 'red': return 'bg-red-500'
      default: return 'bg-gray-500'
    }
  }

  if (loading) {
    return (
      <div className="min-h-screen flex items-center justify-center">
        <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-primary"></div>
      </div>
    )
  }

  if (error) {
    return (
      <div className="py-16">
        <div className="container mx-auto px-4">
          <div className="flex justify-between items-center mb-8">
            <h1 className="text-4xl font-bold">Member Dashboard</h1>
            <button
              onClick={handleLogout}
              className="bg-red-600 text-white px-6 py-2 rounded-lg hover:bg-red-700 transition"
            >
              Logout
            </button>
          </div>
          <div className="bg-red-100 border border-red-200 text-red-800 p-6 rounded-xl">
            <h2 className="text-xl font-bold mb-2">Error Loading Dashboard</h2>
            <p className="mb-4">{error}</p>
            <button
              onClick={fetchMemberStatus}
              className="bg-red-600 text-white px-4 py-2 rounded-lg hover:bg-red-700 transition"
            >
              Retry
            </button>
          </div>
        </div>
      </div>
    )
  }

  if (!memberStatus) {
    return (
      <div className="py-16">
        <div className="container mx-auto px-4">
          <div className="flex justify-between items-center mb-8">
            <h1 className="text-4xl font-bold">Member Dashboard</h1>
            <button
              onClick={handleLogout}
              className="bg-red-600 text-white px-6 py-2 rounded-lg hover:bg-red-700 transition"
            >
              Logout
            </button>
          </div>
          <div className="bg-yellow-100 border border-yellow-200 text-yellow-800 p-6 rounded-xl">
            <h2 className="text-xl font-bold mb-2">No Data Available</h2>
            <p>Unable to load your member information. Please contact support.</p>
          </div>
        </div>
      </div>
    )
  }

  const org = memberStatus?.organization
  const statusInfo = memberStatus?.status_info

  return (
    <div className="py-16">
      <div className="container mx-auto px-4">
        {/* Header */}
        <div className="flex justify-between items-center mb-8">
          <h1 className="text-4xl font-bold">Member Dashboard</h1>
          <button
            onClick={handleLogout}
            className="bg-red-600 text-white px-6 py-2 rounded-lg hover:bg-red-700 transition"
          >
            Logout
          </button>
        </div>

        {/* Status Banner */}
        {statusInfo && (
          <div className={`rounded-xl p-6 mb-8 border-2 ${getStatusColorClass(statusInfo.status_color)}`}>
            <div className="flex items-center gap-3 mb-3">
              <span className={`w-3 h-3 rounded-full ${getStatusBadgeClass(statusInfo.status_color)}`}></span>
              <h2 className="text-xl font-bold">{statusInfo.status_label}</h2>
            </div>
            <p className="text-lg mb-4">{statusInfo.message}</p>
            
            {statusInfo.next_steps.length > 0 && (
              <div className="mt-4">
                <h4 className="font-semibold mb-2">Next Steps:</h4>
                <ul className="list-disc list-inside space-y-1">
                  {statusInfo.next_steps.map((step, index) => (
                    <li key={index}>{step}</li>
                  ))}
                </ul>
              </div>
            )}
          </div>
        )}

        {/* Organization Details Card */}
        {org && (
          <div className="bg-white p-6 rounded-xl shadow-md mb-8 border border-border">
            <div className="flex items-start justify-between mb-6">
              <div className="flex items-center gap-4">
                {org.logo && (
                  <img src={org.logo} alt={org.name} className="w-16 h-16 object-contain rounded-lg border" />
                )}
                <div>
                  <h2 className="text-2xl font-bold">{org.name}</h2>
                  <p className="text-secondary/60">{org.member_type === 'NATIONAL' ? 'National NGO' : 'International NGO'}</p>
                </div>
              </div>
              <div className="flex items-center gap-2">
                {org.is_verified && (
                  <span className="bg-blue-100 text-blue-800 px-3 py-1 rounded-full text-sm font-medium">
                    ✓ Verified
                  </span>
                )}
              </div>
            </div>
            
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
              <div>
                <div className="text-sm text-secondary/60 mb-1">Organization Status</div>
                <div className={`font-semibold ${
                  org.status === 'ACTIVE' ? 'text-green-600' : 
                  org.status === 'PENDING' ? 'text-yellow-600' : 'text-red-600'
                }`}>
                  {org.status}
                </div>
              </div>
              <div>
                <div className="text-sm text-secondary/60 mb-1">Membership Fee</div>
                <div className={`font-semibold ${org.membership_fee_paid ? 'text-green-600' : 'text-red-600'}`}>
                  {org.membership_fee_paid ? 'Paid' : 'Not Paid'}
                </div>
              </div>
              <div>
                <div className="text-sm text-secondary/60 mb-1">Member Since</div>
                <div className="font-semibold">{formatDate(org.date_joined)}</div>
              </div>
              <div>
                <div className="text-sm text-secondary/60 mb-1">Membership Expires</div>
                <div className={`font-semibold ${org.membership_expiry_date ? (isExpired(org.membership_expiry_date) ? 'text-red-600' : 'text-green-600') : ''}`}>
                  {formatDate(org.membership_expiry_date)}
                  {org.membership_expiry_date && isExpired(org.membership_expiry_date) && (
                    <span className="ml-2 text-xs bg-red-100 text-red-700 px-2 py-0.5 rounded-full">Expired</span>
                  )}
                </div>
              </div>
            </div>

            <div className="mt-6 pt-6 border-t border-border grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
              <div>
                <div className="text-sm text-secondary/60 mb-1">Email</div>
                <div className="font-medium">{org.email}</div>
              </div>
              <div>
                <div className="text-sm text-secondary/60 mb-1">Phone</div>
                <div className="font-medium">{org.phone || 'N/A'}</div>
              </div>
              <div>
                <div className="text-sm text-secondary/60 mb-1">Location</div>
                <div className="font-medium">{org.city}, {org.state}</div>
              </div>
            </div>
          </div>
        )}

        {/* Application Details (if available) */}
        {memberStatus?.application && (
          <div className="bg-white p-6 rounded-xl shadow-md mb-8 border border-border">
            <h3 className="text-xl font-bold mb-4">Application Details</h3>
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
              <div>
                <div className="text-sm text-secondary/60 mb-1">Application ID</div>
                <div className="font-mono font-semibold">{memberStatus.application.application_id}</div>
              </div>
              <div>
                <div className="text-sm text-secondary/60 mb-1">Application Status</div>
                <div className={`font-semibold ${
                  memberStatus.application.application_status === 'APPROVED' ? 'text-green-600' :
                  memberStatus.application.application_status === 'PENDING_PAYMENT' ? 'text-yellow-600' :
                  memberStatus.application.application_status === 'REJECTED' ? 'text-red-600' : 'text-blue-600'
                }`}>
                  {memberStatus.application.application_status.replace('_', ' ')}
                </div>
              </div>
              <div>
                <div className="text-sm text-secondary/60 mb-1">Submitted Date</div>
                <div className="font-medium">
                  {memberStatus.application.submitted_date ? 
                    new Date(memberStatus.application.submitted_date).toLocaleDateString() : 'N/A'}
                </div>
              </div>
              <div>
                <div className="text-sm text-secondary/60 mb-1">Reviewed Date</div>
                <div className="font-medium">
                  {memberStatus.application.reviewed_date ? 
                    new Date(memberStatus.application.reviewed_date).toLocaleDateString() : 'Pending'}
                </div>
              </div>
            </div>
          </div>
        )}

        {/* Payment Section - Show only if can pay */}
        {statusInfo?.can_pay && (
          <div className="bg-gradient-to-r from-primary/10 to-primary/5 p-6 rounded-xl mb-8 border-2 border-primary/20">
            <div className="flex items-center gap-3 mb-4">
              <span className="text-3xl">💳</span>
              <h3 className="text-xl font-bold">Complete Your Payment</h3>
            </div>
            
            <p className="text-secondary/80 mb-6">
              Pay your membership fee of <span className="font-bold text-primary">{memberStatus?.currency} {memberStatus?.membership_fee}</span> via MTN Mobile Money to activate your account.
            </p>

            {/* Pending Payments */}
            {memberStatus?.pending_payments && memberStatus.pending_payments.length > 0 && (
              <div className="bg-yellow-50 border border-yellow-200 rounded-lg p-4 mb-6">
                <h4 className="font-semibold text-yellow-800 mb-2">Pending Payment Requests</h4>
                {memberStatus.pending_payments.map((payment, index) => (
                  <div key={index} className="flex items-center justify-between bg-white p-3 rounded-lg mb-2">
                    <div>
                      <div className="text-sm text-secondary/60">Reference: {payment.external_id}</div>
                      <div className="font-semibold">{payment.currency} {payment.amount} - {payment.payer_phone}</div>
                    </div>
                    <button
                      onClick={() => checkPaymentStatus(payment.external_id)}
                      disabled={checkingPayment}
                      className="bg-yellow-500 text-white px-4 py-2 rounded-lg hover:bg-yellow-600 transition disabled:opacity-50"
                    >
                      {checkingPayment ? 'Checking...' : 'Check Status'}
                    </button>
                  </div>
                ))}
              </div>
            )}

            {/* Payment Message */}
            {paymentMessage && (
              <div className={`p-4 rounded-lg mb-6 ${
                paymentMessage.type === 'success' ? 'bg-green-100 text-green-800' : 'bg-red-100 text-red-800'
              }`}>
                {paymentMessage.text}
              </div>
            )}

            {/* Payment Form */}
            <div className="bg-white p-6 rounded-xl">
              <h4 className="font-semibold mb-4">Initiate New Payment</h4>
              <div className="flex flex-col md:flex-row gap-4">
                <div className="flex-1">
                  <label className="block text-sm font-medium text-secondary mb-2">
                    MTN Mobile Money Number
                  </label>
                  <input
                    type="tel"
                    value={phoneNumber}
                    onChange={(e) => setPhoneNumber(e.target.value)}
                    placeholder="e.g., 256771234567 or 0771234567"
                    className="w-full px-4 py-3 border border-border rounded-xl focus:ring-2 focus:ring-primary/20 focus:border-primary outline-none"
                  />
                  <p className="text-sm text-secondary/60 mt-1">
                    Enter the phone number where you want to receive the payment prompt
                  </p>
                </div>
                <div className="flex items-end">
                  <button
                    onClick={initiatePayment}
                    disabled={paymentLoading || !phoneNumber}
                    className="w-full md:w-auto bg-primary text-white px-8 py-3 rounded-xl font-semibold hover:bg-primary/90 transition disabled:opacity-50 disabled:cursor-not-allowed"
                  >
                    {paymentLoading ? (
                      <span className="flex items-center gap-2">
                        <svg className="animate-spin h-5 w-5" viewBox="0 0 24 24">
                          <circle className="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" strokeWidth="4" fill="none"/>
                          <path className="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"/>
                        </svg>
                        Processing...
                      </span>
                    ) : (
                      'Pay Now'
                    )}
                  </button>
                </div>
              </div>
            </div>
          </div>
        )}

        {/* Quick Actions Grid - Only show if active member */}
        {org?.status === 'ACTIVE' && org?.membership_fee_paid && (
          <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
            <Link
              href="/portal/profile"
              className="p-6 border border-border rounded-xl hover:shadow-lg transition bg-white"
            >
              <div className="text-4xl mb-4">👤</div>
              <h3 className="text-xl font-semibold mb-2">Organization Profile</h3>
              <p className="text-secondary/60">Update your organization information and logo</p>
            </Link>

            <Link
              href="/portal/operational-data"
              className="p-6 border border-border rounded-xl hover:shadow-lg transition bg-white"
            >
              <div className="text-4xl mb-4">📊</div>
              <h3 className="text-xl font-semibold mb-2">3W Data</h3>
              <p className="text-secondary/60">Submit and manage operational presence data</p>
            </Link>

            <Link
              href="/portal/documents"
              className="p-6 border border-border rounded-xl hover:shadow-lg transition bg-white"
            >
              <div className="text-4xl mb-4">📄</div>
              <h3 className="text-xl font-semibold mb-2">Documents</h3>
              <p className="text-secondary/60">Upload and manage resources</p>
            </Link>

            <Link
              href="/portal/events"
              className="p-6 border border-border rounded-xl hover:shadow-lg transition bg-white"
            >
              <div className="text-4xl mb-4">📅</div>
              <h3 className="text-xl font-semibold mb-2">Events</h3>
              <p className="text-secondary/60">Post and manage events</p>
            </Link>

            <Link
              href="/portal/jobs"
              className="p-6 border border-border rounded-xl hover:shadow-lg transition bg-white"
            >
              <div className="text-4xl mb-4">💼</div>
              <h3 className="text-xl font-semibold mb-2">Jobs & Tenders</h3>
              <p className="text-secondary/60">Post job openings and tenders</p>
            </Link>

            <Link
              href="/portal/security"
              className="p-6 border border-border rounded-xl hover:shadow-lg transition bg-white"
            >
              <div className="text-4xl mb-4">🔒</div>
              <h3 className="text-xl font-semibold mb-2">Security Reports</h3>
              <p className="text-secondary/60">Report security incidents using 6Ws</p>
            </Link>

            <Link
              href="/forum"
              className="p-6 border border-border rounded-xl hover:shadow-lg transition bg-white"
            >
              <div className="text-4xl mb-4">💬</div>
              <h3 className="text-xl font-semibold mb-2">Forum</h3>
              <p className="text-secondary/60">Participate in discussions</p>
            </Link>
          </div>
        )}

        {/* Limited Access Notice for pending members */}
        {org && (org.status !== 'ACTIVE' || !org.membership_fee_paid) && (
          <div className="bg-gray-100 p-6 rounded-xl border border-gray-200">
            <h3 className="text-xl font-semibold mb-4 text-gray-700">Limited Access</h3>
            <p className="text-gray-600 mb-4">
              Complete your membership payment to unlock full access to:
            </p>
            <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
              <div className="text-center p-4 bg-white rounded-lg opacity-50">
                <div className="text-2xl mb-2">👤</div>
                <div className="text-sm">Profile Management</div>
              </div>
              <div className="text-center p-4 bg-white rounded-lg opacity-50">
                <div className="text-2xl mb-2">📊</div>
                <div className="text-sm">3W Data Submission</div>
              </div>
              <div className="text-center p-4 bg-white rounded-lg opacity-50">
                <div className="text-2xl mb-2">💬</div>
                <div className="text-sm">Forum Discussions</div>
              </div>
              <div className="text-center p-4 bg-white rounded-lg opacity-50">
                <div className="text-2xl mb-2">📄</div>
                <div className="text-sm">Document Access</div>
              </div>
            </div>
          </div>
        )}
      </div>
    </div>
  )
}
