"use client";

import { useState, useEffect } from "react";
import Link from "next/link";

// South Sudan States
const SS_STATES = [
  "Central Equatoria", "Eastern Equatoria", "Western Equatoria",
  "Jonglei", "Unity", "Upper Nile", "Lakes", "Warrap",
  "Western Bahr el Ghazal", "Northern Bahr el Ghazal"
];

// All South Sudan Counties
const SS_COUNTIES = [
  "Abiemnhom", "Abyei", "Akobo", "Akoka", "Aweil Center", "Aweil East",
  "Aweil North", "Aweil South", "Aweil West", "Awerial", "Ayod", "Baliet",
  "Bor", "Budi", "Cueibet", "Duk", "Ezo", "Fangak", "Fashoda", "Gogrial East",
  "Gogrial West", "Guit", "Ibba", "Ikotos", "Jau", "Juba", "Jur River",
  "Kajo Keji", "Kapoeta East", "Kapoeta North", "Kapoeta South", "Koch",
  "Lafon", "Lainya", "Leer", "Longechuk", "Maban", "Magwi", "Maiwut",
  "Malakal", "Manyo", "Maridi", "Mayendit", "Mayom", "Melut", "Morobo",
  "Mundri East", "Mundri West", "Mvolo", "Nagero", "Nasir", "Nyirol",
  "Nzara", "Panriang", "Panyijiar", "Panyikang", "Pibor", "Pigi", "Pochalla",
  "Raja", "Renk", "Rubkona", "Rumbek Center", "Rumbek East", "Rumbek North",
  "Tambura", "Terekeka", "Tonj East", "Tonj North", "Tonj South", "Torit",
  "Twic", "Twic East", "Ulang", "Uror", "Wau", "Wulu", "Yambio", "Yei",
  "Yida", "Yirol East", "Yirol West"
];

// Clusters/Sectors
const CLUSTERS_SECTORS = [
  "Camp Coordination and Camp Management",
  "Education",
  "Emergency Shelter/Non Food Items",
  "Food Security and Livelihoods",
  "Health",
  "Logistics",
  "Nutrition",
  "Protection",
  "Protection - Child Protection",
  "Protection - Gender-based Violence",
  "Infrastructure",
  "Protection - Land Coordination Forum",
  "Protection - Mine Action",
  "Water And Sanitation and Hygiene",
  "Conflict Prevention And Peace Building",
  "Governance And Rule Of Law",
  "Environment And Natural Resources",
  "Media And Journalism",
  "Environment/Climate/Conservation",
  "Economic Development",
  "System Strengthening"
];

// Codes of Conduct options
const CODES_OF_CONDUCT = [
  "Code of Conduct for the Red Cross and NGOs in Disaster Relief",
  "Do No Harm",
  "Humanitarian Accountability Partnership",
  "INEE (Inter-Agency Network for Education in Emergencies)",
  "LEGS (Livestock Emergency Guidelines and Standards)",
  "Organisation's (or alliance's) own Code Of Conduct",
  "People In Aid",
  "Sphere (Humanitarian Charter & Minimum Standards In Humanitarian Response)",
  "Core Humanitarian Standard on Quality and Accountability",
  "None of the above"
];

// Global Networks options
const GLOBAL_NETWORKS = [
  "Alliance2015",
  "ALNAP",
  "ICVA",
  "InterAction",
  "People in Aid",
  "SCHR",
  "SPHERE Standards",
  "VOICE",
  "Charter 4 Change",
  "START Network",
  "Non of the above",
  "None of the above"
];

// Mailing List Groups
const MAILING_GROUPS = [
  "Country Director's Group",
  "National Director's Group",
  "Security Working Group",
  "Finance Working Group",
  "Human Resource Working Group",
  "Localization WG",
  "Durable Solution WG",
  "PSEA WG"
];

// Mailing List Roles
const MAILING_ROLES = [
  "Country Director",
  "Deputy Country Director",
  "Head of Programs",
  "Finance Manager",
  "HR Manager",
  "Security Focal Point",
  "Program Manager",
  "Other"
];

// Countries list (simplified)
const COUNTRIES = [
  "Afghanistan", "Australia", "Austria", "Bangladesh", "Belgium", "Canada",
  "China", "Denmark", "Egypt", "Ethiopia", "Finland", "France", "Germany",
  "India", "Ireland", "Italy", "Japan", "Kenya", "Netherlands", "Norway",
  "Pakistan", "South Africa", "South Sudan", "Spain", "Sudan", "Sweden",
  "Switzerland", "Uganda", "United Kingdom", "United States", "Other"
];

interface FormData {
  // Step 1
  organization_name: string;
  organization_acronym: string;
  organization_type: string;
  organization_brief: string;
  is_non_political: boolean;
  is_non_proselytising: boolean;
  is_humanitarian_actor: boolean;
  has_fulltime_presence: boolean;
  headquarter_country: string;
  ss_main_office_state: string;
  ss_hq_physical_address: string;
  ngo_vehicle_plate_prefix: string;
  year_operations_began: string;
  website: string;
  // Step 2
  rrc_certificate: File | null;
  rrc_registration_number: string;
  rrc_certificate_start_date: string;
  rrc_certificate_expiry_date: string;
  is_woman_led: string;
  international_staff_count: string;
  national_staff_count: string;
  relocatable_staff_count: string;
  juba_residence_compound: string;
  annual_operating_budget_usd: string;
  invoicing_contact_name: string;
  invoicing_contact_email: string;
  invoicing_contact_phone: string;
  // Step 3
  mailing_list_contacts: Array<{
    group: string;
    contact_name: string;
    role: string;
    telephone: string;
    skype_id: string;
    email: string;
  }>;
  // Step 4
  psea_policy: string;
  codes_of_conduct: string[];
  global_networks: string[];
  // Step 5
  clusters_sectors: string[];
  operational_presence: Record<string, string[]>;
}

const initialFormData: FormData = {
  organization_name: "",
  organization_acronym: "",
  organization_type: "",
  organization_brief: "",
  is_non_political: false,
  is_non_proselytising: false,
  is_humanitarian_actor: false,
  has_fulltime_presence: false,
  headquarter_country: "",
  ss_main_office_state: "",
  ss_hq_physical_address: "",
  ngo_vehicle_plate_prefix: "",
  year_operations_began: "",
  website: "",
  rrc_certificate: null,
  rrc_registration_number: "",
  rrc_certificate_start_date: "",
  rrc_certificate_expiry_date: "",
  is_woman_led: "",
  international_staff_count: "",
  national_staff_count: "",
  relocatable_staff_count: "",
  juba_residence_compound: "",
  annual_operating_budget_usd: "",
  invoicing_contact_name: "",
  invoicing_contact_email: "",
  invoicing_contact_phone: "",
  mailing_list_contacts: [{ group: "", contact_name: "", role: "", telephone: "", skype_id: "", email: "" }],
  psea_policy: "",
  codes_of_conduct: [],
  global_networks: [],
  clusters_sectors: [],
  operational_presence: {}
};

export default function RegisterPage() {
  const [currentStep, setCurrentStep] = useState(1);
  const [formData, setFormData] = useState<FormData>(initialFormData);
  const [errors, setErrors] = useState<Record<string, string>>({});
  const [isSubmitting, setIsSubmitting] = useState(false);
  const [submitResult, setSubmitResult] = useState<{ success: boolean; applicationId?: string; message: string } | null>(null);
  const [certificatePreview, setCertificatePreview] = useState<string | null>(null);
  const [isHydrated, setIsHydrated] = useState(false);

  const totalSteps = 5;
  const STORAGE_KEY = 'ngo_registration_form';
  const STEP_KEY = 'ngo_registration_step';

  // Load saved data from localStorage on mount
  useEffect(() => {
    const savedData = localStorage.getItem(STORAGE_KEY);
    const savedStep = localStorage.getItem(STEP_KEY);
    
    if (savedData) {
      try {
        const parsed = JSON.parse(savedData);
        // Don't restore file - it can't be stored in localStorage
        parsed.rrc_certificate = null;
        setFormData(parsed);
      } catch (e) {
        console.error('Failed to parse saved form data');
      }
    }
    
    if (savedStep) {
      const step = parseInt(savedStep, 10);
      if (step >= 1 && step <= totalSteps) {
        setCurrentStep(step);
      }
    }
    
    setIsHydrated(true);
  }, []);

  // Save form data to localStorage whenever it changes
  useEffect(() => {
    if (isHydrated) {
      const dataToSave = { ...formData, rrc_certificate: null }; // Don't save file
      localStorage.setItem(STORAGE_KEY, JSON.stringify(dataToSave));
    }
  }, [formData, isHydrated]);

  // Save current step to localStorage
  useEffect(() => {
    if (isHydrated) {
      localStorage.setItem(STEP_KEY, currentStep.toString());
    }
  }, [currentStep, isHydrated]);

  // Clear saved data after successful submission
  const clearSavedData = () => {
    localStorage.removeItem(STORAGE_KEY);
    localStorage.removeItem(STEP_KEY);
  };

  const updateFormData = (field: keyof FormData, value: any) => {
    setFormData(prev => ({ ...prev, [field]: value }));
    if (errors[field]) {
      setErrors(prev => {
        const newErrors = { ...prev };
        delete newErrors[field];
        return newErrors;
      });
    }
  };

  const handleFileChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const file = e.target.files?.[0];
    if (file) {
      if (file.size > 2 * 1024 * 1024) {
        setErrors(prev => ({ ...prev, rrc_certificate: "File size must be less than 2MB" }));
        return;
      }
      updateFormData("rrc_certificate", file);
      const reader = new FileReader();
      reader.onloadend = () => {
        setCertificatePreview(reader.result as string);
      };
      reader.readAsDataURL(file);
    }
  };

  const addMailingContact = () => {
    setFormData(prev => ({
      ...prev,
      mailing_list_contacts: [...prev.mailing_list_contacts, { group: "", contact_name: "", role: "", telephone: "", skype_id: "", email: "" }]
    }));
  };

  const updateMailingContact = (index: number, field: string, value: string) => {
    setFormData(prev => {
      const contacts = [...prev.mailing_list_contacts];
      contacts[index] = { ...contacts[index], [field]: value };
      return { ...prev, mailing_list_contacts: contacts };
    });
  };

  const removeMailingContact = (index: number) => {
    if (formData.mailing_list_contacts.length > 1) {
      setFormData(prev => ({
        ...prev,
        mailing_list_contacts: prev.mailing_list_contacts.filter((_, i) => i !== index)
      }));
    }
  };

  const toggleCluster = (cluster: string) => {
    setFormData(prev => {
      const clusters = prev.clusters_sectors.includes(cluster)
        ? prev.clusters_sectors.filter(c => c !== cluster)
        : [...prev.clusters_sectors, cluster];
      
      // Also update operational_presence
      const presence = { ...prev.operational_presence };
      if (!clusters.includes(cluster)) {
        delete presence[cluster];
      } else if (!presence[cluster]) {
        presence[cluster] = [];
      }
      
      return { ...prev, clusters_sectors: clusters, operational_presence: presence };
    });
  };

  const toggleCountyForCluster = (cluster: string, county: string) => {
    setFormData(prev => {
      const presence = { ...prev.operational_presence };
      if (!presence[cluster]) {
        presence[cluster] = [];
      }
      if (presence[cluster].includes(county)) {
        presence[cluster] = presence[cluster].filter(c => c !== county);
      } else {
        presence[cluster] = [...presence[cluster], county];
      }
      return { ...prev, operational_presence: presence };
    });
  };

  const validateStep = (step: number): boolean => {
    const newErrors: Record<string, string> = {};

    if (step === 1) {
      if (!formData.organization_name) newErrors.organization_name = "Organization name is required";
      if (!formData.organization_acronym) newErrors.organization_acronym = "Organization acronym is required";
      if (!formData.organization_type) newErrors.organization_type = "Organization type is required";
      if (!formData.organization_brief) newErrors.organization_brief = "Organization brief is required";
      if (!formData.headquarter_country) newErrors.headquarter_country = "Headquarter country is required";
      if (!formData.ss_main_office_state) newErrors.ss_main_office_state = "Main office state is required";
      if (!formData.ss_hq_physical_address) newErrors.ss_hq_physical_address = "Physical address is required";
      if (!formData.year_operations_began) newErrors.year_operations_began = "Year operations began is required";
      if (!formData.website) newErrors.website = "Organization website is required";
    }

    if (step === 2) {
      if (!formData.rrc_certificate) newErrors.rrc_certificate = "RRC Certificate is required";
      if (!formData.rrc_registration_number) newErrors.rrc_registration_number = "RRC Registration number is required";
      if (!formData.rrc_certificate_start_date) newErrors.rrc_certificate_start_date = "RRC start date is required";
      if (!formData.rrc_certificate_expiry_date) newErrors.rrc_certificate_expiry_date = "RRC expiry date is required";
      if (!formData.is_woman_led) newErrors.is_woman_led = "Please indicate if organization is woman-led";
      if (!formData.international_staff_count) newErrors.international_staff_count = "International staff count is required";
      if (!formData.national_staff_count) newErrors.national_staff_count = "National staff count is required";
      if (!formData.relocatable_staff_count) newErrors.relocatable_staff_count = "Relocatable staff count is required";
      if (!formData.invoicing_contact_name) newErrors.invoicing_contact_name = "Invoicing contact name is required";
      if (!formData.invoicing_contact_email) newErrors.invoicing_contact_email = "Invoicing contact email is required";
      if (!formData.invoicing_contact_phone) newErrors.invoicing_contact_phone = "Invoicing contact phone is required";
    }

    if (step === 3) {
      const validContacts = formData.mailing_list_contacts.filter(c => c.group && c.contact_name && c.email);
      if (validContacts.length === 0) {
        newErrors.mailing_list_contacts = "At least one complete mailing list contact is required";
      }
    }

    if (step === 4) {
      if (!formData.psea_policy) newErrors.psea_policy = "PSEA policy status is required";
      if (formData.codes_of_conduct.length === 0) newErrors.codes_of_conduct = "Please select at least one code of conduct";
      if (formData.global_networks.length === 0) newErrors.global_networks = "Please select at least one global network option";
    }

    if (step === 5) {
      if (formData.clusters_sectors.length === 0) {
        newErrors.clusters_sectors = "Please select at least one cluster/sector";
      } else {
        // Check that each selected cluster has at least one county
        for (const cluster of formData.clusters_sectors) {
          if (!formData.operational_presence[cluster] || formData.operational_presence[cluster].length === 0) {
            newErrors.operational_presence = "Please select counties for each selected cluster/sector";
            break;
          }
        }
      }
    }

    setErrors(newErrors);
    return Object.keys(newErrors).length === 0;
  };

  const nextStep = () => {
    if (validateStep(currentStep)) {
      setCurrentStep(prev => Math.min(prev + 1, totalSteps));
      window.scrollTo(0, 0);
    }
  };

  const prevStep = () => {
    setCurrentStep(prev => Math.max(prev - 1, 1));
    window.scrollTo(0, 0);
  };

  const handleSubmit = async () => {
    if (!validateStep(currentStep)) return;

    // Check eligibility
    const eligibilityErrors: string[] = [];
    if (!formData.is_non_political) eligibilityErrors.push("NGO Forum Membership requires members to be Non Political");
    if (!formData.is_non_proselytising) eligibilityErrors.push("NGO Forum Membership requires members to be Non Proselytising");
    if (!formData.is_humanitarian_actor) eligibilityErrors.push("NGO Forum Membership requires members to be Humanitarian actors");
    if (!formData.has_fulltime_presence) eligibilityErrors.push("NGO Forum Membership requires Fulltime Presence in South Sudan");

    if (eligibilityErrors.length > 0) {
      setErrors({ eligibility: eligibilityErrors.join(". ") });
      return;
    }

    setIsSubmitting(true);

    try {
      const submitData = new FormData();
      
      // Add all form fields
      submitData.append("organization_name", formData.organization_name);
      submitData.append("organization_acronym", formData.organization_acronym);
      submitData.append("organization_type", formData.organization_type);
      submitData.append("organization_brief", formData.organization_brief);
      submitData.append("is_non_political", String(formData.is_non_political));
      submitData.append("is_non_proselytising", String(formData.is_non_proselytising));
      submitData.append("is_humanitarian_actor", String(formData.is_humanitarian_actor));
      submitData.append("has_fulltime_presence", String(formData.has_fulltime_presence));
      submitData.append("headquarter_country", formData.headquarter_country);
      submitData.append("ss_main_office_state", formData.ss_main_office_state);
      submitData.append("ss_hq_physical_address", formData.ss_hq_physical_address);
      submitData.append("ngo_vehicle_plate_prefix", formData.ngo_vehicle_plate_prefix);
      submitData.append("year_operations_began", formData.year_operations_began);
      submitData.append("website", formData.website);
      
      if (formData.rrc_certificate) {
        submitData.append("rrc_certificate", formData.rrc_certificate);
      }
      submitData.append("rrc_registration_number", formData.rrc_registration_number);
      submitData.append("rrc_certificate_start_date", formData.rrc_certificate_start_date);
      submitData.append("rrc_certificate_expiry_date", formData.rrc_certificate_expiry_date);
      submitData.append("is_woman_led", formData.is_woman_led);
      submitData.append("international_staff_count", formData.international_staff_count);
      submitData.append("national_staff_count", formData.national_staff_count);
      submitData.append("relocatable_staff_count", formData.relocatable_staff_count);
      submitData.append("juba_residence_compound", formData.juba_residence_compound);
      submitData.append("annual_operating_budget_usd", formData.annual_operating_budget_usd || "0");
      submitData.append("invoicing_contact_name", formData.invoicing_contact_name);
      submitData.append("invoicing_contact_email", formData.invoicing_contact_email);
      submitData.append("invoicing_contact_phone", formData.invoicing_contact_phone);
      
      submitData.append("mailing_list_contacts", JSON.stringify(formData.mailing_list_contacts.filter(c => c.group && c.contact_name)));
      submitData.append("psea_policy", formData.psea_policy);
      submitData.append("codes_of_conduct", JSON.stringify(formData.codes_of_conduct));
      submitData.append("global_networks", JSON.stringify(formData.global_networks));
      submitData.append("clusters_sectors", JSON.stringify(formData.clusters_sectors));
      submitData.append("operational_presence", JSON.stringify(formData.operational_presence));

      const API_URL = process.env.NEXT_PUBLIC_API_URL || "http://localhost:8000/api";
      const response = await fetch(`${API_URL}/applications/`, {
        method: "POST",
        body: submitData,
      });

      const result = await response.json();
      console.log("API Response:", response.status, result);

      if (response.ok) {
        clearSavedData(); // Clear saved form data on success
        setSubmitResult({
          success: true,
          applicationId: result.application_id,
          message: result.message || "Application submitted successfully!"
        });
      } else {
        // Extract error message from various possible response formats
        let errorMessage = "Failed to submit application. Please try again.";
        if (result.message) {
          errorMessage = result.message;
        } else if (result.detail) {
          errorMessage = result.detail;
        } else if (result.eligibility) {
          errorMessage = Array.isArray(result.eligibility) ? result.eligibility.join(". ") : result.eligibility;
        } else if (typeof result === 'object') {
          // Handle field-level validation errors
          const errorFields = Object.keys(result);
          if (errorFields.length > 0) {
            const fieldErrors = errorFields.map(field => {
              const fieldError = result[field];
              return `${field}: ${Array.isArray(fieldError) ? fieldError.join(", ") : fieldError}`;
            });
            errorMessage = fieldErrors.join(". ");
          }
        }
        
        setSubmitResult({
          success: false,
          message: errorMessage
        });
      }
    } catch (error) {
      console.error("Submission error:", error);
      setSubmitResult({
        success: false,
        message: "An error occurred while submitting your application. Please check your internet connection and try again."
      });
    } finally {
      setIsSubmitting(false);
    }
  };

  // Loading state while hydrating from localStorage
  if (!isHydrated) {
    return (
      <div className="min-h-screen bg-background flex items-center justify-center">
        <div className="text-center">
          <div className="w-12 h-12 border-4 border-primary border-t-transparent rounded-full animate-spin mx-auto mb-4" />
          <p className="text-secondary/60">Loading your progress...</p>
        </div>
      </div>
    );
  }

  // Success screen
  if (submitResult?.success) {
    return (
      <div className="min-h-screen bg-background py-12">
        <div className="max-w-2xl mx-auto px-4">
          <div className="bg-white rounded-3xl border border-border p-8 md:p-12 text-center">
            <div className="w-20 h-20 bg-green-100 rounded-full flex items-center justify-center mx-auto mb-6">
              <svg className="w-10 h-10 text-green-500" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M5 13l4 4L19 7" />
              </svg>
            </div>
            
            <h1 className="text-3xl font-bold text-secondary mb-4">Application Submitted!</h1>
            <p className="text-secondary/70 mb-6">{submitResult.message}</p>
            
            <div className="bg-primary/5 rounded-2xl p-6 mb-8">
              <p className="text-sm text-secondary/60 mb-2">Your Application ID</p>
              <p className="text-2xl font-bold text-primary font-mono">{submitResult.applicationId}</p>
              <p className="text-sm text-secondary/60 mt-2">Please save this ID to track your application status</p>
            </div>
            
            <div className="space-y-4">
              <div className="p-4 bg-blue-50 rounded-xl text-left">
                <h3 className="font-semibold text-blue-800 mb-2">What happens next?</h3>
                <ol className="text-sm text-blue-700 space-y-2 list-decimal list-inside">
                  <li>NGO Forum Secretariat will review your application</li>
                  <li>Your application will be presented to the Steering Committee</li>
                  <li>Final vote by Country Directors group</li>
                  <li>You will be notified of the decision via email</li>
                </ol>
              </div>
              
              <Link
                href="/"
                className="inline-flex items-center justify-center px-8 py-4 rounded-2xl bg-primary text-white font-semibold hover:bg-primary-light transition-colors"
              >
                Return to Homepage
              </Link>
            </div>
          </div>
        </div>
      </div>
    );
  }

  // Error screen
  if (submitResult && !submitResult.success) {
    return (
      <div className="min-h-screen bg-background py-12">
        <div className="max-w-2xl mx-auto px-4">
          <div className="bg-white rounded-3xl border border-border p-8 md:p-12 text-center">
            <div className="w-20 h-20 bg-red-100 rounded-full flex items-center justify-center mx-auto mb-6">
              <svg className="w-10 h-10 text-red-500" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M6 18L18 6M6 6l12 12" />
              </svg>
            </div>
            
            <h1 className="text-3xl font-bold text-secondary mb-4">Submission Failed</h1>
            <p className="text-secondary/70 mb-6">{submitResult.message}</p>
            
            <div className="space-y-4">
              <div className="p-4 bg-amber-50 rounded-xl text-left">
                <h3 className="font-semibold text-amber-800 mb-2">What to do next?</h3>
                <ul className="text-sm text-amber-700 space-y-2 list-disc list-inside">
                  <li>Check your internet connection</li>
                  <li>Verify all required fields are filled correctly</li>
                  <li>Try submitting again</li>
                  <li>If the problem persists, contact support</li>
                </ul>
              </div>
              
              <div className="flex flex-col sm:flex-row gap-4 justify-center">
                <button
                  onClick={() => setSubmitResult(null)}
                  className="inline-flex items-center justify-center px-8 py-4 rounded-2xl bg-primary text-white font-semibold hover:bg-primary-light transition-colors"
                >
                  Try Again
                </button>
                <Link
                  href="/contact"
                  className="inline-flex items-center justify-center px-8 py-4 rounded-2xl border-2 border-secondary/20 text-secondary font-semibold hover:bg-secondary/5 transition-colors"
                >
                  Contact Support
                </Link>
              </div>
            </div>
          </div>
        </div>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-background">
      {/* Header */}
      <div className="bg-secondary text-white py-6">
        <div className="max-w-5xl mx-auto px-4">
          <div className="flex items-center justify-between">
            <div className="flex items-center gap-4">
              <Link href="/apply" className="text-white/80 hover:text-white">
                <svg className="w-6 h-6" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M10 19l-7-7m0 0l7-7m-7 7h18" />
                </svg>
              </Link>
              <div>
                <h1 className="text-xl font-bold">NGO Registration</h1>
                <p className="text-sm text-white/70">South Sudan NGO Forum</p>
              </div>
            </div>
            <button
              onClick={() => {
                if (confirm('Are you sure you want to clear all form data and start over?')) {
                  clearSavedData();
                  setFormData(initialFormData);
                  setCurrentStep(1);
                  setCertificatePreview(null);
                }
              }}
              className="text-sm text-white/60 hover:text-white transition-colors flex items-center gap-2"
            >
              <svg className="w-4 h-4" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M4 4v5h.582m15.356 2A8.001 8.001 0 004.582 9m0 0H9m11 11v-5h-.581m0 0a8.003 8.003 0 01-15.357-2m15.357 2H15" />
              </svg>
              Start Over
            </button>
          </div>
        </div>
      </div>

      {/* Progress Bar */}
      <div className="bg-white border-b border-border py-4">
        <div className="max-w-5xl mx-auto px-4">
          <div className="flex gap-2">
            {[1, 2, 3, 4, 5].map((step) => (
              <div
                key={step}
                className={`flex-1 h-2 rounded-full transition-colors ${
                  step <= currentStep ? "bg-primary" : "bg-gray-200"
                }`}
              />
            ))}
          </div>
          <div className="mt-3 flex justify-between text-sm">
            <span className={currentStep === 1 ? "text-primary font-medium" : "text-secondary/50"}>Basic Info</span>
            <span className={currentStep === 2 ? "text-primary font-medium" : "text-secondary/50"}>Other Info</span>
            <span className={currentStep === 3 ? "text-primary font-medium" : "text-secondary/50"}>Mailing List</span>
            <span className={currentStep === 4 ? "text-primary font-medium" : "text-secondary/50"}>Code of Conduct</span>
            <span className={currentStep === 5 ? "text-primary font-medium" : "text-secondary/50"}>Operations</span>
          </div>
        </div>
      </div>

      {/* Form Content */}
      <div className="max-w-5xl mx-auto px-4 py-8">
        {errors.eligibility && (
          <div className="mb-6 p-4 bg-red-50 border border-red-200 rounded-xl">
            <div className="flex items-start gap-3">
              <svg className="w-6 h-6 text-red-500 flex-shrink-0" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-3L13.732 4c-.77-1.333-2.694-1.333-3.464 0L3.34 16c-.77 1.333.192 3 1.732 3z" />
              </svg>
              <div>
                <p className="font-semibold text-red-800">Eligibility Requirements Not Met</p>
                <p className="text-red-700 text-sm mt-1">{errors.eligibility}</p>
              </div>
            </div>
          </div>
        )}

        <div className="bg-white rounded-2xl border border-border p-6 md:p-8">
          {/* Step 1: Organization Basic Information */}
          {currentStep === 1 && (
            <div>
              <h2 className="text-xl font-bold text-secondary mb-6 pb-4 border-b border-border">
                ORGANIZATION BASIC INFORMATION
              </h2>

              <div className="space-y-6">
                <div className="grid md:grid-cols-2 gap-6">
                  <div>
                    <label className="block text-sm font-medium text-secondary mb-2">
                      Organization Name <span className="text-red-500">*</span>
                    </label>
                    <input
                      type="text"
                      value={formData.organization_name}
                      onChange={(e) => updateFormData("organization_name", e.target.value)}
                      className={`w-full px-4 py-3 border rounded-xl focus:ring-2 focus:ring-primary/20 focus:border-primary outline-none transition-colors ${errors.organization_name ? "border-red-500" : "border-border"}`}
                      placeholder="Name of your Organization"
                    />
                    {errors.organization_name && <p className="text-red-500 text-sm mt-1">{errors.organization_name}</p>}
                  </div>

                  <div>
                    <label className="block text-sm font-medium text-secondary mb-2">
                      Organization Acronym <span className="text-red-500">*</span>
                    </label>
                    <input
                      type="text"
                      value={formData.organization_acronym}
                      onChange={(e) => updateFormData("organization_acronym", e.target.value)}
                      className={`w-full px-4 py-3 border rounded-xl focus:ring-2 focus:ring-primary/20 focus:border-primary outline-none transition-colors ${errors.organization_acronym ? "border-red-500" : "border-border"}`}
                      placeholder="e.g., RRC, NGO-X"
                    />
                    {errors.organization_acronym && <p className="text-red-500 text-sm mt-1">{errors.organization_acronym}</p>}
                  </div>
                </div>

                <div>
                  <label className="block text-sm font-medium text-secondary mb-2">
                    Organization Type <span className="text-red-500">*</span>
                  </label>
                  <select
                    value={formData.organization_type}
                    onChange={(e) => updateFormData("organization_type", e.target.value)}
                    className={`w-full px-4 py-3 border rounded-xl focus:ring-2 focus:ring-primary/20 focus:border-primary outline-none transition-colors ${errors.organization_type ? "border-red-500" : "border-border"}`}
                  >
                    <option value="">Select organization type</option>
                    <option value="INGO">International NGO (INGO)</option>
                    <option value="NNGO">National NGO (NNGO)</option>
                    <option value="OBSERVER">Observer</option>
                  </select>
                  {errors.organization_type && <p className="text-red-500 text-sm mt-1">{errors.organization_type}</p>}
                </div>

                <div>
                  <label className="block text-sm font-medium text-secondary mb-2">
                    Short brief of your organization <span className="text-red-500">*</span>
                  </label>
                  <textarea
                    value={formData.organization_brief}
                    onChange={(e) => updateFormData("organization_brief", e.target.value)}
                    rows={4}
                    className={`w-full px-4 py-3 border rounded-xl focus:ring-2 focus:ring-primary/20 focus:border-primary outline-none transition-colors ${errors.organization_brief ? "border-red-500" : "border-border"}`}
                    placeholder="Example: MY-ORGANISATION is a humanitarian and development organization that is changing the way people think about and deliver aid in Africa..."
                  />
                  {errors.organization_brief && <p className="text-red-500 text-sm mt-1">{errors.organization_brief}</p>}
                </div>

                {/* Eligibility Questions */}
                <div className="bg-amber-50 border border-amber-200 rounded-xl p-6">
                  <h3 className="font-semibold text-amber-800 mb-4">Eligibility Requirements</h3>
                  
                  <div className="space-y-4">
                    <div className="flex items-center justify-between p-3 bg-white rounded-lg">
                      <span className="text-secondary">Non Political <span className="text-red-500">*</span></span>
                      <div className="flex gap-4">
                        <label className="flex items-center gap-2 cursor-pointer">
                          <input
                            type="radio"
                            name="is_non_political"
                            checked={formData.is_non_political === true}
                            onChange={() => updateFormData("is_non_political", true)}
                            className="w-4 h-4 text-primary"
                          />
                          <span className="text-green-600">Yes</span>
                        </label>
                        <label className="flex items-center gap-2 cursor-pointer">
                          <input
                            type="radio"
                            name="is_non_political"
                            checked={formData.is_non_political === false}
                            onChange={() => updateFormData("is_non_political", false)}
                            className="w-4 h-4 text-primary"
                          />
                          <span className="text-red-600">No</span>
                        </label>
                      </div>
                    </div>

                    <div className="flex items-center justify-between p-3 bg-white rounded-lg">
                      <span className="text-secondary">Non Proselytising <span className="text-red-500">*</span></span>
                      <div className="flex gap-4">
                        <label className="flex items-center gap-2 cursor-pointer">
                          <input
                            type="radio"
                            name="is_non_proselytising"
                            checked={formData.is_non_proselytising === true}
                            onChange={() => updateFormData("is_non_proselytising", true)}
                            className="w-4 h-4 text-primary"
                          />
                          <span className="text-green-600">Yes</span>
                        </label>
                        <label className="flex items-center gap-2 cursor-pointer">
                          <input
                            type="radio"
                            name="is_non_proselytising"
                            checked={formData.is_non_proselytising === false}
                            onChange={() => updateFormData("is_non_proselytising", false)}
                            className="w-4 h-4 text-primary"
                          />
                          <span className="text-red-600">No</span>
                        </label>
                      </div>
                    </div>

                    <div className="flex items-center justify-between p-3 bg-white rounded-lg">
                      <span className="text-secondary">Humanitarian And Development Assistance Actor <span className="text-red-500">*</span></span>
                      <div className="flex gap-4">
                        <label className="flex items-center gap-2 cursor-pointer">
                          <input
                            type="radio"
                            name="is_humanitarian_actor"
                            checked={formData.is_humanitarian_actor === true}
                            onChange={() => updateFormData("is_humanitarian_actor", true)}
                            className="w-4 h-4 text-primary"
                          />
                          <span className="text-green-600">Yes</span>
                        </label>
                        <label className="flex items-center gap-2 cursor-pointer">
                          <input
                            type="radio"
                            name="is_humanitarian_actor"
                            checked={formData.is_humanitarian_actor === false}
                            onChange={() => updateFormData("is_humanitarian_actor", false)}
                            className="w-4 h-4 text-primary"
                          />
                          <span className="text-red-600">No</span>
                        </label>
                      </div>
                    </div>

                    <div className="flex items-center justify-between p-3 bg-white rounded-lg">
                      <span className="text-secondary">Full Time Presence in South Sudan <span className="text-red-500">*</span></span>
                      <div className="flex gap-4">
                        <label className="flex items-center gap-2 cursor-pointer">
                          <input
                            type="radio"
                            name="has_fulltime_presence"
                            checked={formData.has_fulltime_presence === true}
                            onChange={() => updateFormData("has_fulltime_presence", true)}
                            className="w-4 h-4 text-primary"
                          />
                          <span className="text-green-600">Yes</span>
                        </label>
                        <label className="flex items-center gap-2 cursor-pointer">
                          <input
                            type="radio"
                            name="has_fulltime_presence"
                            checked={formData.has_fulltime_presence === false}
                            onChange={() => updateFormData("has_fulltime_presence", false)}
                            className="w-4 h-4 text-primary"
                          />
                          <span className="text-red-600">No</span>
                        </label>
                      </div>
                    </div>
                  </div>
                </div>

                <div className="grid md:grid-cols-2 gap-6">
                  <div>
                    <label className="block text-sm font-medium text-secondary mb-2">
                      Headquarter Country <span className="text-red-500">*</span>
                    </label>
                    <select
                      value={formData.headquarter_country}
                      onChange={(e) => updateFormData("headquarter_country", e.target.value)}
                      className={`w-full px-4 py-3 border rounded-xl focus:ring-2 focus:ring-primary/20 focus:border-primary outline-none transition-colors ${errors.headquarter_country ? "border-red-500" : "border-border"}`}
                    >
                      <option value="">Select country</option>
                      {COUNTRIES.map(country => (
                        <option key={country} value={country}>{country}</option>
                      ))}
                    </select>
                    {errors.headquarter_country && <p className="text-red-500 text-sm mt-1">{errors.headquarter_country}</p>}
                  </div>

                  <div>
                    <label className="block text-sm font-medium text-secondary mb-2">
                      State Where South Sudan Main Office is Located <span className="text-red-500">*</span>
                    </label>
                    <select
                      value={formData.ss_main_office_state}
                      onChange={(e) => updateFormData("ss_main_office_state", e.target.value)}
                      className={`w-full px-4 py-3 border rounded-xl focus:ring-2 focus:ring-primary/20 focus:border-primary outline-none transition-colors ${errors.ss_main_office_state ? "border-red-500" : "border-border"}`}
                    >
                      <option value="">Select state</option>
                      {SS_STATES.map(state => (
                        <option key={state} value={state}>{state}</option>
                      ))}
                    </select>
                    {errors.ss_main_office_state && <p className="text-red-500 text-sm mt-1">{errors.ss_main_office_state}</p>}
                  </div>
                </div>

                <div>
                  <label className="block text-sm font-medium text-secondary mb-2">
                    South Sudan Headquarters Physical Address <span className="text-red-500">*</span>
                  </label>
                  <input
                    type="text"
                    value={formData.ss_hq_physical_address}
                    onChange={(e) => updateFormData("ss_hq_physical_address", e.target.value)}
                    className={`w-full px-4 py-3 border rounded-xl focus:ring-2 focus:ring-primary/20 focus:border-primary outline-none transition-colors ${errors.ss_hq_physical_address ? "border-red-500" : "border-border"}`}
                    placeholder="e.g., Hai Jerusalem, Juba, South Sudan"
                  />
                  {errors.ss_hq_physical_address && <p className="text-red-500 text-sm mt-1">{errors.ss_hq_physical_address}</p>}
                </div>

                <div className="grid md:grid-cols-2 gap-6">
                  <div>
                    <label className="block text-sm font-medium text-secondary mb-2">
                      NGO Vehicle(s) Number Plate Prefix
                    </label>
                    <input
                      type="text"
                      value={formData.ngo_vehicle_plate_prefix}
                      onChange={(e) => updateFormData("ngo_vehicle_plate_prefix", e.target.value)}
                      className="w-full px-4 py-3 border border-border rounded-xl focus:ring-2 focus:ring-primary/20 focus:border-primary outline-none transition-colors"
                      placeholder="e.g., NGO 09"
                    />
                  </div>

                  <div>
                    <label className="block text-sm font-medium text-secondary mb-2">
                      Year Operations Began in South Sudan <span className="text-red-500">*</span>
                    </label>
                    <input
                      type="number"
                      value={formData.year_operations_began}
                      onChange={(e) => updateFormData("year_operations_began", e.target.value)}
                      className={`w-full px-4 py-3 border rounded-xl focus:ring-2 focus:ring-primary/20 focus:border-primary outline-none transition-colors ${errors.year_operations_began ? "border-red-500" : "border-border"}`}
                      placeholder="e.g., 2015"
                      min="1900"
                      max={new Date().getFullYear()}
                    />
                    {errors.year_operations_began && <p className="text-red-500 text-sm mt-1">{errors.year_operations_began}</p>}
                  </div>
                </div>

                <div>
                  <label className="block text-sm font-medium text-secondary mb-2">
                    Organization Website <span className="text-red-500">*</span>
                  </label>
                  <input
                    type="url"
                    value={formData.website}
                    onChange={(e) => updateFormData("website", e.target.value)}
                    className={`w-full px-4 py-3 border rounded-xl focus:ring-2 focus:ring-primary/20 focus:border-primary outline-none transition-colors ${errors.website ? 'border-red-500' : 'border-border'}`}
                    placeholder="https://www.example.org"
                    required
                  />
                  {errors.website && <p className="text-red-500 text-sm mt-1">{errors.website}</p>}
                </div>
              </div>
            </div>
          )}

          {/* Step 2: Organization Other Information */}
          {currentStep === 2 && (
            <div>
              <h2 className="text-xl font-bold text-secondary mb-6 pb-4 border-b border-border">
                ORGANIZATION OTHER INFORMATION
              </h2>

              <div className="space-y-6">
                <div>
                  <label className="block text-sm font-medium text-secondary mb-2">
                    Upload RRC Certificate <span className="text-red-500">*</span>
                  </label>
                  <p className="text-sm text-secondary/60 mb-3">Upload a picture of your Organization&apos;s RRC Certificate (2MB max, jpeg/png)</p>
                  
                  <div className="flex items-start gap-6">
                    <div className="w-36 h-36 bg-gray-100 border-2 border-dashed border-border rounded-xl flex items-center justify-center overflow-hidden">
                      {certificatePreview ? (
                        <img src={certificatePreview} alt="Certificate preview" className="w-full h-full object-cover" />
                      ) : (
                        <span className="text-gray-400 text-sm text-center px-2">150 x 150</span>
                      )}
                    </div>
                    <div>
                      <input
                        type="file"
                        accept=".jpg,.jpeg,.png"
                        onChange={handleFileChange}
                        className="hidden"
                        id="rrc_certificate"
                      />
                      <label
                        htmlFor="rrc_certificate"
                        className="inline-flex items-center gap-2 px-4 py-2 bg-secondary text-white rounded-lg cursor-pointer hover:bg-secondary-dark transition-colors"
                      >
                        <svg className="w-5 h-5" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                          <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M4 16l4.586-4.586a2 2 0 012.828 0L16 16m-2-2l1.586-1.586a2 2 0 012.828 0L20 14m-6-6h.01M6 20h12a2 2 0 002-2V6a2 2 0 00-2-2H6a2 2 0 00-2 2v12a2 2 0 002 2z" />
                        </svg>
                        Browse
                      </label>
                      {formData.rrc_certificate && (
                        <p className="text-sm text-green-600 mt-2">✓ {formData.rrc_certificate.name}</p>
                      )}
                    </div>
                  </div>
                  {errors.rrc_certificate && <p className="text-red-500 text-sm mt-2">{errors.rrc_certificate}</p>}
                </div>

                <div className="grid md:grid-cols-3 gap-6">
                  <div>
                    <label className="block text-sm font-medium text-secondary mb-2">
                      RRC Registration Number <span className="text-red-500">*</span>
                    </label>
                    <input
                      type="text"
                      value={formData.rrc_registration_number}
                      onChange={(e) => updateFormData("rrc_registration_number", e.target.value)}
                      className={`w-full px-4 py-3 border rounded-xl focus:ring-2 focus:ring-primary/20 focus:border-primary outline-none transition-colors ${errors.rrc_registration_number ? "border-red-500" : "border-border"}`}
                      placeholder="As indicated on RRC Certificate"
                    />
                    {errors.rrc_registration_number && <p className="text-red-500 text-sm mt-1">{errors.rrc_registration_number}</p>}
                  </div>

                  <div>
                    <label className="block text-sm font-medium text-secondary mb-2">
                      RRC Certificate Start Date <span className="text-red-500">*</span>
                    </label>
                    <input
                      type="date"
                      value={formData.rrc_certificate_start_date}
                      onChange={(e) => updateFormData("rrc_certificate_start_date", e.target.value)}
                      className={`w-full px-4 py-3 border rounded-xl focus:ring-2 focus:ring-primary/20 focus:border-primary outline-none transition-colors ${errors.rrc_certificate_start_date ? "border-red-500" : "border-border"}`}
                    />
                    {errors.rrc_certificate_start_date && <p className="text-red-500 text-sm mt-1">{errors.rrc_certificate_start_date}</p>}
                  </div>

                  <div>
                    <label className="block text-sm font-medium text-secondary mb-2">
                      RRC Certificate Expiry Date <span className="text-red-500">*</span>
                    </label>
                    <input
                      type="date"
                      value={formData.rrc_certificate_expiry_date}
                      onChange={(e) => updateFormData("rrc_certificate_expiry_date", e.target.value)}
                      className={`w-full px-4 py-3 border rounded-xl focus:ring-2 focus:ring-primary/20 focus:border-primary outline-none transition-colors ${errors.rrc_certificate_expiry_date ? "border-red-500" : "border-border"}`}
                    />
                    {errors.rrc_certificate_expiry_date && <p className="text-red-500 text-sm mt-1">{errors.rrc_certificate_expiry_date}</p>}
                  </div>
                </div>

                <div>
                  <label className="block text-sm font-medium text-secondary mb-2">
                    Is Your Organization a Woman-Led Organization? <span className="text-red-500">*</span>
                  </label>
                  <div className="flex gap-4">
                    <label className="flex items-center gap-2 cursor-pointer">
                      <input
                        type="radio"
                        name="is_woman_led"
                        checked={formData.is_woman_led === "YES"}
                        onChange={() => updateFormData("is_woman_led", "YES")}
                        className="w-4 h-4 text-primary"
                      />
                      <span>Yes</span>
                    </label>
                    <label className="flex items-center gap-2 cursor-pointer">
                      <input
                        type="radio"
                        name="is_woman_led"
                        checked={formData.is_woman_led === "NO"}
                        onChange={() => updateFormData("is_woman_led", "NO")}
                        className="w-4 h-4 text-primary"
                      />
                      <span>No</span>
                    </label>
                  </div>
                  {errors.is_woman_led && <p className="text-red-500 text-sm mt-1">{errors.is_woman_led}</p>}
                </div>

                <div className="grid md:grid-cols-3 gap-6">
                  <div>
                    <label className="block text-sm font-medium text-secondary mb-2">
                      Estimated Number of International Staff <span className="text-red-500">*</span>
                    </label>
                    <input
                      type="number"
                      value={formData.international_staff_count}
                      onChange={(e) => updateFormData("international_staff_count", e.target.value)}
                      className={`w-full px-4 py-3 border rounded-xl focus:ring-2 focus:ring-primary/20 focus:border-primary outline-none transition-colors ${errors.international_staff_count ? "border-red-500" : "border-border"}`}
                      min="0"
                    />
                    {errors.international_staff_count && <p className="text-red-500 text-sm mt-1">{errors.international_staff_count}</p>}
                  </div>

                  <div>
                    <label className="block text-sm font-medium text-secondary mb-2">
                      Estimated Number of National Staff <span className="text-red-500">*</span>
                    </label>
                    <input
                      type="number"
                      value={formData.national_staff_count}
                      onChange={(e) => updateFormData("national_staff_count", e.target.value)}
                      className={`w-full px-4 py-3 border rounded-xl focus:ring-2 focus:ring-primary/20 focus:border-primary outline-none transition-colors ${errors.national_staff_count ? "border-red-500" : "border-border"}`}
                      min="0"
                    />
                    {errors.national_staff_count && <p className="text-red-500 text-sm mt-1">{errors.national_staff_count}</p>}
                  </div>

                  <div>
                    <label className="block text-sm font-medium text-secondary mb-2">
                      Estimated Number of Relocatable Staff <span className="text-red-500">*</span>
                    </label>
                    <input
                      type="number"
                      value={formData.relocatable_staff_count}
                      onChange={(e) => updateFormData("relocatable_staff_count", e.target.value)}
                      className={`w-full px-4 py-3 border rounded-xl focus:ring-2 focus:ring-primary/20 focus:border-primary outline-none transition-colors ${errors.relocatable_staff_count ? "border-red-500" : "border-border"}`}
                      min="0"
                    />
                    {errors.relocatable_staff_count && <p className="text-red-500 text-sm mt-1">{errors.relocatable_staff_count}</p>}
                  </div>
                </div>

                <div className="grid md:grid-cols-2 gap-6">
                  <div>
                    <label className="block text-sm font-medium text-secondary mb-2">
                      Juba Residence Compound
                    </label>
                    <input
                      type="text"
                      value={formData.juba_residence_compound}
                      onChange={(e) => updateFormData("juba_residence_compound", e.target.value)}
                      className="w-full px-4 py-3 border border-border rounded-xl focus:ring-2 focus:ring-primary/20 focus:border-primary outline-none transition-colors"
                      placeholder="Juba Residence Compound"
                    />
                  </div>

                  <div>
                    <label className="block text-sm font-medium text-secondary mb-2">
                      Annual Operating Budget for 2026 (USD)
                    </label>
                    <input
                      type="number"
                      value={formData.annual_operating_budget_usd}
                      onChange={(e) => updateFormData("annual_operating_budget_usd", e.target.value)}
                      className="w-full px-4 py-3 border border-border rounded-xl focus:ring-2 focus:ring-primary/20 focus:border-primary outline-none transition-colors"
                      placeholder="Please convert to USD"
                      min="0"
                    />
                  </div>
                </div>

                <div className="border-t border-border pt-6">
                  <h3 className="font-semibold text-secondary mb-4">Invoicing Contact for Membership Fee Payment</h3>
                  
                  <div className="grid md:grid-cols-3 gap-6">
                    <div>
                      <label className="block text-sm font-medium text-secondary mb-2">
                        Contact Person Name <span className="text-red-500">*</span>
                      </label>
                      <input
                        type="text"
                        value={formData.invoicing_contact_name}
                        onChange={(e) => updateFormData("invoicing_contact_name", e.target.value)}
                        className={`w-full px-4 py-3 border rounded-xl focus:ring-2 focus:ring-primary/20 focus:border-primary outline-none transition-colors ${errors.invoicing_contact_name ? "border-red-500" : "border-border"}`}
                      />
                      {errors.invoicing_contact_name && <p className="text-red-500 text-sm mt-1">{errors.invoicing_contact_name}</p>}
                    </div>

                    <div>
                      <label className="block text-sm font-medium text-secondary mb-2">
                        Contact Person Email <span className="text-red-500">*</span>
                      </label>
                      <input
                        type="email"
                        value={formData.invoicing_contact_email}
                        onChange={(e) => updateFormData("invoicing_contact_email", e.target.value)}
                        className={`w-full px-4 py-3 border rounded-xl focus:ring-2 focus:ring-primary/20 focus:border-primary outline-none transition-colors ${errors.invoicing_contact_email ? "border-red-500" : "border-border"}`}
                        placeholder="xyz@myngoname.org"
                      />
                      {errors.invoicing_contact_email && <p className="text-red-500 text-sm mt-1">{errors.invoicing_contact_email}</p>}
                    </div>

                    <div>
                      <label className="block text-sm font-medium text-secondary mb-2">
                        Contact Person Telephone <span className="text-red-500">*</span>
                      </label>
                      <input
                        type="tel"
                        value={formData.invoicing_contact_phone}
                        onChange={(e) => updateFormData("invoicing_contact_phone", e.target.value)}
                        className={`w-full px-4 py-3 border rounded-xl focus:ring-2 focus:ring-primary/20 focus:border-primary outline-none transition-colors ${errors.invoicing_contact_phone ? "border-red-500" : "border-border"}`}
                        placeholder="+211 916 000 000"
                      />
                      {errors.invoicing_contact_phone && <p className="text-red-500 text-sm mt-1">{errors.invoicing_contact_phone}</p>}
                    </div>
                  </div>
                </div>
              </div>
            </div>
          )}

          {/* Step 3: Mailing List Information */}
          {currentStep === 3 && (
            <div>
              <h2 className="text-xl font-bold text-secondary mb-6 pb-4 border-b border-border">
                MAILING LIST INFORMATION
              </h2>

              <div className="bg-blue-50 border border-blue-200 rounded-xl p-4 mb-6">
                <p className="text-blue-800 text-sm">
                  The NGO Forum has various mailing lists (Country Director&apos;s Group, National Director&apos;s Group, Security Working Group, Finance Working Group, Human Resource Working Group, Localization WG, Durable Solution WG, PSEA WG) to share valued information for the wider good of the NGOs. We require that you update these details often due to the large turn over in the organizations. <strong>Please add a maximum of 2 people per mailing list group.</strong>
                </p>
              </div>

              <div className="space-y-4">
                {formData.mailing_list_contacts.map((contact, index) => (
                  <div key={index} className="border border-border rounded-xl p-4">
                    <div className="flex justify-between items-center mb-4">
                      <span className="font-medium text-secondary">Contact {index + 1}</span>
                      {formData.mailing_list_contacts.length > 1 && (
                        <button
                          type="button"
                          onClick={() => removeMailingContact(index)}
                          className="text-red-500 hover:text-red-700 text-sm"
                        >
                          Remove
                        </button>
                      )}
                    </div>
                    
                    <div className="grid md:grid-cols-3 gap-4">
                      <div>
                        <label className="block text-sm font-medium text-secondary mb-2">
                          Group <span className="text-red-500">*</span>
                        </label>
                        <select
                          value={contact.group}
                          onChange={(e) => updateMailingContact(index, "group", e.target.value)}
                          className="w-full px-4 py-3 border border-border rounded-xl focus:ring-2 focus:ring-primary/20 focus:border-primary outline-none transition-colors"
                        >
                          <option value="">Select group</option>
                          {MAILING_GROUPS.map(group => (
                            <option key={group} value={group}>{group}</option>
                          ))}
                        </select>
                      </div>

                      <div>
                        <label className="block text-sm font-medium text-secondary mb-2">
                          Contact Name <span className="text-red-500">*</span>
                        </label>
                        <input
                          type="text"
                          value={contact.contact_name}
                          onChange={(e) => updateMailingContact(index, "contact_name", e.target.value)}
                          className="w-full px-4 py-3 border border-border rounded-xl focus:ring-2 focus:ring-primary/20 focus:border-primary outline-none transition-colors"
                        />
                      </div>

                      <div>
                        <label className="block text-sm font-medium text-secondary mb-2">
                          Role <span className="text-red-500">*</span>
                        </label>
                        <select
                          value={contact.role}
                          onChange={(e) => updateMailingContact(index, "role", e.target.value)}
                          className="w-full px-4 py-3 border border-border rounded-xl focus:ring-2 focus:ring-primary/20 focus:border-primary outline-none transition-colors"
                        >
                          <option value="">Select role</option>
                          {MAILING_ROLES.map(role => (
                            <option key={role} value={role}>{role}</option>
                          ))}
                        </select>
                      </div>

                      <div>
                        <label className="block text-sm font-medium text-secondary mb-2">
                          Telephone (Preferably WhatsApp) <span className="text-red-500">*</span>
                        </label>
                        <input
                          type="tel"
                          value={contact.telephone}
                          onChange={(e) => updateMailingContact(index, "telephone", e.target.value)}
                          className="w-full px-4 py-3 border border-border rounded-xl focus:ring-2 focus:ring-primary/20 focus:border-primary outline-none transition-colors"
                        />
                      </div>

                      <div>
                        <label className="block text-sm font-medium text-secondary mb-2">
                          Skype ID
                        </label>
                        <input
                          type="text"
                          value={contact.skype_id}
                          onChange={(e) => updateMailingContact(index, "skype_id", e.target.value)}
                          className="w-full px-4 py-3 border border-border rounded-xl focus:ring-2 focus:ring-primary/20 focus:border-primary outline-none transition-colors"
                        />
                      </div>

                      <div>
                        <label className="block text-sm font-medium text-secondary mb-2">
                          Email Address <span className="text-red-500">*</span>
                        </label>
                        <input
                          type="email"
                          value={contact.email}
                          onChange={(e) => updateMailingContact(index, "email", e.target.value)}
                          className="w-full px-4 py-3 border border-border rounded-xl focus:ring-2 focus:ring-primary/20 focus:border-primary outline-none transition-colors"
                        />
                      </div>
                    </div>
                  </div>
                ))}

                <button
                  type="button"
                  onClick={addMailingContact}
                  className="inline-flex items-center gap-2 text-primary hover:text-primary-dark font-medium"
                >
                  <svg className="w-5 h-5" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 6v6m0 0v6m0-6h6m-6 0H6" />
                  </svg>
                  Add another person
                </button>
              </div>
              {errors.mailing_list_contacts && <p className="text-red-500 text-sm mt-4">{errors.mailing_list_contacts}</p>}
            </div>
          )}

          {/* Step 4: Code of Conducts */}
          {currentStep === 4 && (
            <div>
              <h2 className="text-xl font-bold text-secondary mb-6 pb-4 border-b border-border">
                CODE OF CONDUCTS
              </h2>

              <div className="space-y-8">
                <div>
                  <label className="block text-sm font-medium text-secondary mb-4">
                    Does Your Organisation Have A PSEA Policy And Reporting Line? <span className="text-red-500">*</span>
                  </label>
                  <p className="text-sm text-secondary/60 mb-3">Does Your Organisation Have A Prevention Of Sexual Exploitation And Abuse(PSEA) Policy And Reporting Line?</p>
                  
                  <div className="space-y-2">
                    {[
                      { value: "YES", label: "Yes" },
                      { value: "NO", label: "No" },
                      { value: "WORKING", label: "We are working on it" },
                      { value: "UNKNOWN", label: "I Don't Know" }
                    ].map(option => (
                      <label key={option.value} className="flex items-center gap-3 p-3 border border-border rounded-lg cursor-pointer hover:bg-gray-50 transition-colors">
                        <input
                          type="radio"
                          name="psea_policy"
                          checked={formData.psea_policy === option.value}
                          onChange={() => updateFormData("psea_policy", option.value)}
                          className="w-4 h-4 text-primary"
                        />
                        <span>{option.label}</span>
                      </label>
                    ))}
                  </div>
                  {errors.psea_policy && <p className="text-red-500 text-sm mt-2">{errors.psea_policy}</p>}
                </div>

                <div>
                  <label className="block text-sm font-medium text-secondary mb-4">
                    Which of the following Codes Of Conduct is your NGO signatory to? <span className="text-red-500">*</span>
                  </label>
                  
                  <div className="space-y-2">
                    {CODES_OF_CONDUCT.map(code => (
                      <label key={code} className="flex items-center gap-3 p-3 border border-border rounded-lg cursor-pointer hover:bg-gray-50 transition-colors">
                        <input
                          type="checkbox"
                          checked={formData.codes_of_conduct.includes(code)}
                          onChange={() => {
                            const updated = formData.codes_of_conduct.includes(code)
                              ? formData.codes_of_conduct.filter(c => c !== code)
                              : [...formData.codes_of_conduct, code];
                            updateFormData("codes_of_conduct", updated);
                          }}
                          className="w-4 h-4 text-primary rounded"
                        />
                        <span>{code}</span>
                      </label>
                    ))}
                  </div>
                  {errors.codes_of_conduct && <p className="text-red-500 text-sm mt-2">{errors.codes_of_conduct}</p>}
                </div>

                <div>
                  <label className="block text-sm font-medium text-secondary mb-4">
                    Which of the following Global Networks are you part of? <span className="text-red-500">*</span>
                  </label>
                  
                  <div className="space-y-2">
                    {GLOBAL_NETWORKS.map(network => (
                      <label key={network} className="flex items-center gap-3 p-3 border border-border rounded-lg cursor-pointer hover:bg-gray-50 transition-colors">
                        <input
                          type="checkbox"
                          checked={formData.global_networks.includes(network)}
                          onChange={() => {
                            const updated = formData.global_networks.includes(network)
                              ? formData.global_networks.filter(n => n !== network)
                              : [...formData.global_networks, network];
                            updateFormData("global_networks", updated);
                          }}
                          className="w-4 h-4 text-primary rounded"
                        />
                        <span>{network}</span>
                      </label>
                    ))}
                  </div>
                  {errors.global_networks && <p className="text-red-500 text-sm mt-2">{errors.global_networks}</p>}
                </div>
              </div>
            </div>
          )}

          {/* Step 5: Operational Presence */}
          {currentStep === 5 && (
            <div>
              <h2 className="text-xl font-bold text-secondary mb-6 pb-4 border-b border-border">
                ORGANIZATION OPERATIONAL PRESENCE INFORMATION
              </h2>

              <div className="bg-blue-50 border border-blue-200 rounded-xl p-4 mb-6">
                <p className="text-blue-800 text-sm font-medium mb-2">NOTE: Operational Presence is defined as:</p>
                <p className="text-blue-700 text-sm">Organisations that are physically present in the county and are delivering a humanitarian/Development service or implementing a humanitarian/Development project.</p>
                <p className="text-blue-600 text-sm mt-2 italic">To ensure consistency with other datasets (RRC, OCHA and Monthly Cluster Data)&apos;s please ONLY tick areas that you are delivering an activity/project.</p>
              </div>

              <div className="space-y-6">
                <div>
                  <label className="block text-sm font-medium text-secondary mb-4">
                    Which cluster/sector/s is your NGO implementing a humanitarian/Development project? <span className="text-red-500">*</span>
                  </label>
                  
                  <div className="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-3">
                    {CLUSTERS_SECTORS.map(cluster => (
                      <label key={cluster} className={`flex items-center gap-2 p-3 border rounded-lg cursor-pointer transition-colors ${formData.clusters_sectors.includes(cluster) ? "border-primary bg-primary/5" : "border-border hover:bg-gray-50"}`}>
                        <input
                          type="checkbox"
                          checked={formData.clusters_sectors.includes(cluster)}
                          onChange={() => toggleCluster(cluster)}
                          className="w-4 h-4 text-primary rounded"
                        />
                        <span className="text-sm">{cluster}</span>
                      </label>
                    ))}
                  </div>
                  {errors.clusters_sectors && <p className="text-red-500 text-sm mt-2">{errors.clusters_sectors}</p>}
                </div>

                {formData.clusters_sectors.length > 0 && (
                  <div className="bg-amber-50 border border-amber-200 rounded-xl p-4">
                    <p className="text-amber-800 font-medium mb-2">IMPORTANT:</p>
                    <p className="text-amber-700 text-sm">The data in this section below will be used to generate 3Ws Operational Presence data visualizations at county level.</p>
                    <p className="text-amber-700 text-sm mt-2">NOTE: Select all the counties that you are delivering the above activities / that your organization is physically present.</p>
                    <p className="text-red-600 text-sm mt-2 font-medium italic">Failure to select the correct counties may lead to your organization being left out on opportunities when the forum generates reports on members implementing projects in these areas.</p>
                  </div>
                )}

                {formData.clusters_sectors.map(cluster => (
                  <div key={cluster} className="border border-border rounded-xl p-4">
                    <h4 className="font-semibold text-secondary mb-4">
                      {cluster} Cluster/Sector <span className="text-red-500">*</span>
                    </h4>
                    <p className="text-sm text-secondary/60 mb-4">{cluster}</p>
                    
                    <div className="grid grid-cols-3 md:grid-cols-5 lg:grid-cols-7 gap-2">
                      {SS_COUNTIES.map(county => (
                        <label key={county} className={`flex items-center gap-2 p-2 border rounded-lg cursor-pointer text-sm transition-colors ${formData.operational_presence[cluster]?.includes(county) ? "border-primary bg-primary/5" : "border-border hover:bg-gray-50"}`}>
                          <input
                            type="checkbox"
                            checked={formData.operational_presence[cluster]?.includes(county) || false}
                            onChange={() => toggleCountyForCluster(cluster, county)}
                            className="w-3 h-3 text-primary rounded"
                          />
                          <span className="truncate">{county}</span>
                        </label>
                      ))}
                    </div>
                  </div>
                ))}
                {errors.operational_presence && <p className="text-red-500 text-sm mt-2">{errors.operational_presence}</p>}
              </div>
            </div>
          )}

          {/* Navigation Buttons */}
          <div className="flex justify-between mt-8 pt-6 border-t border-border">
            {currentStep > 1 ? (
              <button
                type="button"
                onClick={prevStep}
                className="px-6 py-3 border border-border rounded-xl text-secondary hover:bg-gray-50 transition-colors"
              >
                Previous
              </button>
            ) : (
              <Link
                href="/apply"
                className="px-6 py-3 border border-border rounded-xl text-secondary hover:bg-gray-50 transition-colors"
              >
                Back to Requirements
              </Link>
            )}

            {currentStep < totalSteps ? (
              <button
                type="button"
                onClick={nextStep}
                className="px-8 py-3 bg-green-500 text-white rounded-xl hover:bg-green-600 transition-colors font-medium"
              >
                Next
              </button>
            ) : (
              <button
                type="button"
                onClick={handleSubmit}
                disabled={isSubmitting}
                className="px-8 py-3 bg-green-500 text-white rounded-xl hover:bg-green-600 transition-colors font-medium disabled:opacity-50 disabled:cursor-not-allowed flex items-center gap-2"
              >
                {isSubmitting ? (
                  <>
                    <svg className="animate-spin w-5 h-5" fill="none" viewBox="0 0 24 24">
                      <circle className="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" strokeWidth="4" />
                      <path className="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z" />
                    </svg>
                    Submitting...
                  </>
                ) : (
                  "Submit Application"
                )}
              </button>
            )}
          </div>
        </div>
      </div>
    </div>
  );
}
