import Link from "next/link";

export default function ApplyPage() {
  const benefits = [
    {
      icon: "📢",
      title: "Information Sharing & Communication",
      description: "Access our unique information network ensuring effective, transparent sharing of information relevant to members and external stakeholders."
    },
    {
      icon: "🤝",
      title: "Coordination",
      description: "Coordinate NGO activities through various meetings and relationships with external stakeholders."
    },
    {
      icon: "📋",
      title: "Representation & Advocacy",
      description: "We represent member interests in external meetings, policy discussions, advocacy meetings, and lobbying on issues of common interest."
    },
    {
      icon: "🛡️",
      title: "Safety & Security",
      description: "Receive accessible and timely security and access-related information, advice, and best practices."
    },
    {
      icon: "🌱",
      title: "Networking & Local NGO Development",
      description: "Benefit from capacity building programs and engagement with INGOs and donors through our specialized Local NGO Forum."
    }
  ];

  const eligibilityCriteria = [
    "Nongovernmental, not-for-profit organization",
    "Non-partisan and non-political",
    "Non-proselytising organization",
    "Full-time presence in South Sudan",
    "Appropriate registration with Ministry of Justice and RRC",
    "Willingness to pay membership fees",
    "Subject to approval by NGO Forum Steering Committee and Country Directors group"
  ];

  const applicationProcess = [
    {
      step: 1,
      title: "Complete Application Form",
      description: "Fill out the online application form with all required organization details and documentation."
    },
    {
      step: 2,
      title: "Secretariat Review",
      description: "NGO Forum Secretariat reviews your organization details for completeness and accuracy."
    },
    {
      step: 3,
      title: "Steering Committee Vetting",
      description: "Present a brief description of your organization and activities for vetting/endorsement by the Steering Committee (meets every 2 weeks)."
    },
    {
      step: 4,
      title: "Country Directors Vote",
      description: "Present to the monthly Country Directors group, after which membership is voted upon."
    }
  ];

  const requiredDocuments = {
    basic: [
      { field: "Organization Name", required: true },
      { field: "Organization Acronym/Short Name", required: true },
      { field: "Organization Type (National/International)", required: true },
      { field: "Short Brief of Your Organization", required: true },
      { field: "RRC Certificate Serial Number & Expiry Date", required: true },
      { field: "RRC Certificate Upload (PDF or Image)", required: true },
      { field: "Year Operations Began in South Sudan", required: true },
      { field: "Headquarter Country", required: true },
      { field: "Postal Address", required: true },
      { field: "Website", required: false }
    ],
    contact: [
      { field: "Organization Representative & Contacts", required: true },
      { field: "Alternate Representative & Contacts", required: true },
      { field: "Operating Budgets", required: true },
      { field: "Number of International Staff", required: true },
      { field: "Number of National Staff", required: true },
      { field: "Number of Relocatable Staff", required: true },
      { field: "South Sudan HQ Physical Address", required: true },
      { field: "South Sudan Office Location", required: true },
      { field: "NGO Vehicle Number Plate", required: false }
    ],
    partnerships: [
      { field: "Ministries with Signed MOU", required: false },
      { field: "Ministries without Signed MOU", required: false },
      { field: "PSEA Policy and Reporting Lines", required: false },
      { field: "Codes of Conduct", required: false },
      { field: "Global Networks", required: false }
    ],
    operational: [
      { field: "Sector/Cluster", required: true },
      { field: "County of Operation", required: true },
      { field: "Project Intended Start Year", required: true }
    ]
  };

  return (
    <div className="min-h-screen bg-background">
      {/* Hero Section */}
      <section className="relative overflow-hidden">
        <div className="absolute inset-0 bg-gradient-to-br from-secondary via-secondary-dark to-secondary" />
        <div className="absolute inset-0 bg-gradient-to-b from-transparent via-transparent to-background" />
        <div className="absolute top-1/4 right-1/4 w-96 h-96 bg-primary/20 rounded-full blur-3xl animate-pulse" />
        <div className="absolute bottom-1/4 left-1/4 w-80 h-80 bg-white/10 rounded-full blur-3xl animate-pulse" style={{ animationDelay: '1s' }} />
        
        <div className="relative max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-20 lg:py-28">
          <div className="text-center max-w-4xl mx-auto">
            <div className="inline-flex items-center gap-2 px-4 py-2 rounded-full bg-white/10 border border-white/20 mb-8">
              <span className="w-2 h-2 rounded-full bg-green-500 animate-pulse" />
              <span className="text-sm text-white/80">Membership Registration</span>
            </div>
            
            <h1 className="text-4xl sm:text-5xl lg:text-6xl font-bold tracking-tight mb-6">
              <span className="text-white">Membership</span>
              <span className="bg-gradient-to-r from-primary to-primary-light bg-clip-text text-transparent block mt-2">Registration Process</span>
            </h1>
            
            <p className="text-xl text-white/80 max-w-2xl mx-auto">
              Review the requirements and prepare your documents before starting your application. 
              This ensures a smooth registration process.
            </p>
          </div>
        </div>
      </section>

      {/* Who Can Join Section */}
      <section className="py-16 bg-background">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="max-w-4xl mx-auto">
            <div className="bg-white rounded-3xl border border-border p-8 md:p-12 shadow-sm">
              <div className="flex items-start gap-4 mb-6">
                <div className="w-14 h-14 rounded-2xl bg-primary/10 flex items-center justify-center flex-shrink-0">
                  <svg className="w-7 h-7 text-primary" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M17 20h5v-2a3 3 0 00-5.356-1.857M17 20H7m10 0v-2c0-.656-.126-1.283-.356-1.857M7 20H2v-2a3 3 0 015.356-1.857M7 20v-2c0-.656.126-1.283.356-1.857m0 0a5.002 5.002 0 019.288 0M15 7a3 3 0 11-6 0 3 3 0 016 0zm6 3a2 2 0 11-4 0 2 2 0 014 0zM7 10a2 2 0 11-4 0 2 2 0 014 0z" />
                  </svg>
                </div>
                <div>
                  <h2 className="text-2xl font-bold text-secondary mb-2">Who Can Be a Member?</h2>
                  <p className="text-secondary/70">The following criteria must be met to qualify for membership</p>
                </div>
              </div>
              
              <div className="grid gap-3">
                {eligibilityCriteria.map((criteria, index) => (
                  <div key={index} className="flex items-start gap-3 p-4 rounded-xl bg-background border border-border/50">
                    <div className="flex-shrink-0 w-6 h-6 rounded-full bg-green-500/20 flex items-center justify-center mt-0.5">
                      <svg className="w-4 h-4 text-green-500" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M5 13l4 4L19 7" />
                      </svg>
                    </div>
                    <span className="text-secondary">{criteria}</span>
                  </div>
                ))}
              </div>

              <div className="mt-8 p-4 rounded-xl bg-amber-50 border border-amber-200">
                <div className="flex items-start gap-3">
                  <svg className="w-6 h-6 text-amber-600 flex-shrink-0 mt-0.5" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-3L13.732 4c-.77-1.333-2.694-1.333-3.464 0L3.34 16c-.77 1.333.192 3 1.732 3z" />
                  </svg>
                  <div>
                    <p className="font-semibold text-amber-800 mb-1">Not Eligible</p>
                    <p className="text-amber-700 text-sm">International organisations, businesses, NGOs without full-time presence in South Sudan, or for-profit NGOs may not apply for membership.</p>
                  </div>
                </div>
              </div>
            </div>
          </div>
        </div>
      </section>

      {/* Benefits Section */}
      <section className="py-16 bg-white">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="text-center mb-12">
            <h2 className="text-3xl font-bold mb-4 text-secondary">
              Member <span className="bg-gradient-to-r from-primary to-primary-light bg-clip-text text-transparent">Benefits</span>
            </h2>
            <p className="text-secondary/60 max-w-2xl mx-auto">
              What you gain by becoming a member of the South Sudan NGO Forum
            </p>
          </div>

          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
            {benefits.map((benefit, index) => (
              <div
                key={index}
                className="group p-6 rounded-2xl bg-background border border-border hover:border-primary/30 transition-all duration-300 hover:shadow-lg hover:shadow-primary/5"
              >
                <div className="text-3xl mb-3">{benefit.icon}</div>
                <h3 className="text-lg font-semibold mb-2 text-secondary group-hover:text-primary transition-colors">
                  {benefit.title}
                </h3>
                <p className="text-secondary/60 text-sm leading-relaxed">
                  {benefit.description}
                </p>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* Application Process Section */}
      <section className="py-16 bg-background">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="text-center mb-12">
            <h2 className="text-3xl font-bold mb-4 text-secondary">
              Application <span className="bg-gradient-to-r from-primary to-primary-light bg-clip-text text-transparent">Process</span>
            </h2>
            <p className="text-secondary/60 max-w-2xl mx-auto">
              How your organization can apply for membership
            </p>
          </div>

          <div className="max-w-4xl mx-auto">
            <div className="relative">
              {/* Timeline line */}
              <div className="absolute left-8 top-0 bottom-0 w-0.5 bg-border hidden md:block" />
              
              <div className="space-y-6">
                {applicationProcess.map((step, index) => (
                  <div key={index} className="relative flex gap-6 items-start">
                    <div className="relative z-10 flex-shrink-0 w-16 h-16 rounded-2xl bg-primary/10 flex items-center justify-center border-4 border-background">
                      <span className="text-2xl font-bold text-primary">{step.step}</span>
                    </div>
                    <div className="flex-1 pt-3">
                      <h3 className="text-xl font-semibold text-secondary mb-2">{step.title}</h3>
                      <p className="text-secondary/60">{step.description}</p>
                    </div>
                  </div>
                ))}
              </div>
            </div>
          </div>

          <div className="max-w-4xl mx-auto mt-10">
            <div className="p-6 rounded-2xl bg-blue-50 border border-blue-200">
              <div className="flex items-start gap-4">
                <svg className="w-6 h-6 text-blue-600 flex-shrink-0 mt-0.5" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M13 16h-1v-4h-1m1-4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z" />
                </svg>
                <div>
                  <p className="font-semibold text-blue-800 mb-1">Observer Member Status</p>
                  <p className="text-blue-700 text-sm">In certain circumstances, organizations can apply for observer membership which allows use of forum services but without voting rights. Observer membership applications must be accompanied by a letter explaining the reasons for seeking observer status, which is then reviewed by the Steering Committee.</p>
                </div>
              </div>
            </div>
          </div>
        </div>
      </section>

      {/* Required Documents Section */}
      <section className="py-16 bg-white">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="text-center mb-12">
            <h2 className="text-3xl font-bold mb-4 text-secondary">
              Required <span className="bg-gradient-to-r from-primary to-primary-light bg-clip-text text-transparent">Information</span>
            </h2>
            <p className="text-secondary/60 max-w-2xl mx-auto">
              Please prepare the following information before starting your application. 
              Fields marked with <span className="text-red-500 font-semibold">*</span> are required.
            </p>
          </div>

          <div className="max-w-5xl mx-auto grid gap-6 md:grid-cols-2">
            {/* Basic Information */}
            <div className="bg-background rounded-2xl border border-border p-6">
              <div className="flex items-center gap-3 mb-6">
                <div className="w-10 h-10 rounded-xl bg-primary/10 flex items-center justify-center">
                  <svg className="w-5 h-5 text-primary" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 21V5a2 2 0 00-2-2H7a2 2 0 00-2 2v16m14 0h2m-2 0h-5m-9 0H3m2 0h5M9 7h1m-1 4h1m4-4h1m-1 4h1m-5 10v-5a1 1 0 011-1h2a1 1 0 011 1v5m-4 0h4" />
                  </svg>
                </div>
                <h3 className="text-lg font-semibold text-secondary">Organization Basic Information</h3>
              </div>
              <ul className="space-y-2">
                {requiredDocuments.basic.map((doc, index) => (
                  <li key={index} className="flex items-start gap-2 text-sm">
                    <span className="text-secondary/60">•</span>
                    <span className="text-secondary">{doc.field}</span>
                    {doc.required && <span className="text-red-500">*</span>}
                  </li>
                ))}
              </ul>
            </div>

            {/* Contact & Staffing */}
            <div className="bg-background rounded-2xl border border-border p-6">
              <div className="flex items-center gap-3 mb-6">
                <div className="w-10 h-10 rounded-xl bg-primary/10 flex items-center justify-center">
                  <svg className="w-5 h-5 text-primary" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M17 20h5v-2a3 3 0 00-5.356-1.857M17 20H7m10 0v-2c0-.656-.126-1.283-.356-1.857M7 20H2v-2a3 3 0 015.356-1.857M7 20v-2c0-.656.126-1.283.356-1.857m0 0a5.002 5.002 0 019.288 0M15 7a3 3 0 11-6 0 3 3 0 016 0z" />
                  </svg>
                </div>
                <h3 className="text-lg font-semibold text-secondary">Contact & Staffing Information</h3>
              </div>
              <ul className="space-y-2">
                {requiredDocuments.contact.map((doc, index) => (
                  <li key={index} className="flex items-start gap-2 text-sm">
                    <span className="text-secondary/60">•</span>
                    <span className="text-secondary">{doc.field}</span>
                    {doc.required && <span className="text-red-500">*</span>}
                  </li>
                ))}
              </ul>
            </div>

            {/* Partnerships & Codes */}
            <div className="bg-background rounded-2xl border border-border p-6">
              <div className="flex items-center gap-3 mb-6">
                <div className="w-10 h-10 rounded-xl bg-primary/10 flex items-center justify-center">
                  <svg className="w-5 h-5 text-primary" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z" />
                  </svg>
                </div>
                <h3 className="text-lg font-semibold text-secondary">Partnerships & Code of Conduct</h3>
              </div>
              <ul className="space-y-2">
                {requiredDocuments.partnerships.map((doc, index) => (
                  <li key={index} className="flex items-start gap-2 text-sm">
                    <span className="text-secondary/60">•</span>
                    <span className="text-secondary">{doc.field}</span>
                    {doc.required && <span className="text-red-500">*</span>}
                  </li>
                ))}
              </ul>
            </div>

            {/* Operational Presence */}
            <div className="bg-background rounded-2xl border border-border p-6">
              <div className="flex items-center gap-3 mb-6">
                <div className="w-10 h-10 rounded-xl bg-primary/10 flex items-center justify-center">
                  <svg className="w-5 h-5 text-primary" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M17.657 16.657L13.414 20.9a1.998 1.998 0 01-2.827 0l-4.244-4.243a8 8 0 1111.314 0z" />
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M15 11a3 3 0 11-6 0 3 3 0 016 0z" />
                  </svg>
                </div>
                <h3 className="text-lg font-semibold text-secondary">Operational Presence</h3>
              </div>
              <ul className="space-y-2">
                {requiredDocuments.operational.map((doc, index) => (
                  <li key={index} className="flex items-start gap-2 text-sm">
                    <span className="text-secondary/60">•</span>
                    <span className="text-secondary">{doc.field}</span>
                    {doc.required && <span className="text-red-500">*</span>}
                  </li>
                ))}
              </ul>
            </div>
          </div>
        </div>
      </section>

      {/* Membership Period Info */}
      <section className="py-16 bg-background">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="max-w-4xl mx-auto">
            <div className="bg-white rounded-3xl border border-border p-8 md:p-12 shadow-sm">
              <div className="flex items-start gap-4 mb-6">
                <div className="w-14 h-14 rounded-2xl bg-primary/10 flex items-center justify-center flex-shrink-0">
                  <svg className="w-7 h-7 text-primary" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M8 7V3m8 4V3m-9 8h10M5 21h14a2 2 0 002-2V7a2 2 0 00-2-2H5a2 2 0 00-2 2v12a2 2 0 002 2z" />
                  </svg>
                </div>
                <div>
                  <h2 className="text-2xl font-bold text-secondary mb-2">Membership Period</h2>
                  <p className="text-secondary/70">Important information about membership renewal</p>
                </div>
              </div>
              
              <div className="space-y-4 text-secondary/80">
                <p>
                  Membership renewal/payment is done on a <span className="font-semibold text-secondary">yearly basis</span>. 
                  Renewal notifications are usually sent to members in January.
                </p>
                <p>
                  Renewal requires completion of the same application form as first-time members to ensure your organization&apos;s information is up to date.
                </p>
              </div>

              <div className="mt-8 flex items-center gap-3 p-4 rounded-xl bg-primary/5 border border-primary/10">
                <span className="text-3xl">💰</span>
                <div>
                  <p className="font-semibold text-secondary">Annual Membership Fee: $200</p>
                  <p className="text-secondary/60 text-sm">Less than $17/month for complete access to all benefits</p>
                </div>
              </div>
            </div>
          </div>
        </div>
      </section>

      {/* CTA Section */}
      <section className="relative py-20 overflow-hidden">
        <div className="absolute inset-0 bg-gradient-to-br from-secondary via-secondary-dark to-secondary" />
        <div className="absolute inset-0 bg-gradient-to-t from-secondary/50 to-transparent" />
        <div className="absolute top-1/4 left-1/4 w-96 h-96 bg-primary/20 rounded-full blur-3xl animate-pulse" />
        
        <div className="relative max-w-4xl mx-auto px-4 sm:px-6 lg:px-8 text-center">
          <div className="mb-8">
            <div className="inline-flex items-center gap-2 px-4 py-2 rounded-full bg-white/10 border border-white/20 mb-6">
              <svg className="w-5 h-5 text-green-400" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z" />
              </svg>
              <span className="text-sm text-white/80">I have read and understood the requirements</span>
            </div>
          </div>
          
          <h2 className="text-3xl sm:text-4xl lg:text-5xl font-bold mb-6 text-white">
            Ready to <span className="bg-gradient-to-r from-primary to-primary-light bg-clip-text text-transparent">Apply?</span>
          </h2>
          <p className="text-xl text-white/80 mb-10 max-w-2xl mx-auto">
            Once you&apos;ve prepared all the required documents, proceed to complete your registration application.
          </p>
          
          <Link
            href="/apply/register"
            className="inline-flex items-center justify-center px-10 py-5 text-xl font-semibold rounded-full bg-primary text-white hover:bg-primary-light transition-all duration-300 shadow-xl shadow-primary/30 hover:shadow-primary/50 hover:scale-105"
          >
            Proceed to Application Form
            <svg className="ml-3 w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M13 7l5 5m0 0l-5 5m5-5H6" />
            </svg>
          </Link>
          
          <p className="mt-6 text-white/50 text-sm">
            Complete the multi-step registration form
          </p>

          <div className="mt-10 pt-10 border-t border-white/10">
            <p className="text-white/60 text-sm mb-4">Have questions about membership?</p>
            <Link
              href="/faqs"
              className="inline-flex items-center gap-2 text-white/80 hover:text-white transition-colors"
            >
              View Frequently Asked Questions
              <svg className="w-4 h-4" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 5l7 7-7 7" />
              </svg>
            </Link>
          </div>
        </div>
      </section>
    </div>
  );
}
