from rest_framework import viewsets, permissions, filters, status
from rest_framework.decorators import action, api_view, permission_classes as perm_decorator
from rest_framework.response import Response
from rest_framework.parsers import MultiPartParser, FormParser, JSONParser
from rest_framework.views import APIView
from django_filters.rest_framework import DjangoFilterBackend
from django.views.decorators.csrf import csrf_exempt
from django.utils.decorators import method_decorator
from django.utils import timezone
from django.core.mail import send_mail
from django.conf import settings
from datetime import date
import json
import uuid
from .models import MemberOrganization, OrganizationContact, StaffMember, MembershipApplication, MembershipPayment, MoMoPaymentRequest
from .serializers import (
    MemberOrganizationListSerializer, MemberOrganizationDetailSerializer,
    MemberOrganizationWriteSerializer, OrganizationContactSerializer,
    StaffMemberSerializer, MembershipApplicationSerializer, MembershipApplicationCreateSerializer,
    MembershipPaymentSerializer
)


class IsOwnerOrReadOnly(permissions.BasePermission):
    """Custom permission to only allow owners to edit their organization"""
    def has_object_permission(self, request, view, obj):
        if request.method in permissions.SAFE_METHODS:
            return True
        return obj.user == request.user if hasattr(obj, 'user') else False


class PublicMemberViewSet(viewsets.ReadOnlyModelViewSet):
    """Public read-only access to member organizations"""
    queryset = MemberOrganization.objects.filter(status='ACTIVE')
    permission_classes = [permissions.AllowAny]
    filter_backends = [DjangoFilterBackend, filters.SearchFilter, filters.OrderingFilter]
    filterset_fields = ['member_type', 'state', 'city', 'is_verified']
    search_fields = ['name', 'description', 'state', 'city']
    ordering_fields = ['name', 'date_joined', 'member_type']
    ordering = ['name']
    lookup_field = 'slug'
    
    def get_serializer_class(self):
        if self.action == 'retrieve':
            return MemberOrganizationDetailSerializer
        return MemberOrganizationListSerializer


class MemberProfileViewSet(viewsets.ModelViewSet):
    """Authenticated member profile management"""
    permission_classes = [permissions.IsAuthenticated, IsOwnerOrReadOnly]
    
    def get_queryset(self):
        # Members can only access their own organization
        if hasattr(self.request.user, 'member_organization'):
            return MemberOrganization.objects.filter(id=self.request.user.member_organization.id)
        return MemberOrganization.objects.none()
    
    def get_serializer_class(self):
        if self.request.method in ['PUT', 'PATCH']:
            return MemberOrganizationWriteSerializer
        return MemberOrganizationDetailSerializer
    
    @action(detail=True, methods=['post'])
    def add_contact(self, request, pk=None):
        """Add a contact person to the organization"""
        organization = self.get_object()
        serializer = OrganizationContactSerializer(data=request.data)
        
        if serializer.is_valid():
            serializer.save(organization=organization)
            return Response(serializer.data, status=status.HTTP_201_CREATED)
        return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)


class MemberStatusView(APIView):
    """
    Get complete member status including:
    - Organization details
    - Application status
    - Payment status
    - Next steps for the member
    """
    permission_classes = [permissions.IsAuthenticated]
    
    def get(self, request):
        user = request.user
        
        # Check if user has an organization
        if hasattr(user, 'member_organization') and user.member_organization:
            org = user.member_organization
            
            # Get the application linked to this organization
            application = None
            try:
                application = MembershipApplication.objects.get(approved_organization=org)
            except MembershipApplication.DoesNotExist:
                pass
            
            # Get any pending MoMo payment requests
            pending_payments = []
            if application:
                pending_payments = MoMoPaymentRequest.objects.filter(
                    application=application,
                    status='PENDING'
                ).values('external_id', 'amount', 'currency', 'payer_phone', 'requested_at')
            
            # Determine the overall status and next steps
            status_info = self._get_status_info(org, application)
            
            return Response({
                'success': True,
                'has_organization': True,
                'organization': {
                    'id': org.id,
                    'name': org.name,
                    'slug': org.slug,
                    'member_type': org.member_type,
                    'email': org.email,
                    'phone': org.phone,
                    'website': org.website,
                    'address': org.address,
                    'city': org.city,
                    'state': org.state,
                    'description': org.description,
                    'logo': org.logo.url if org.logo else None,
                    'status': org.status,
                    'is_verified': org.is_verified,
                    'membership_fee_paid': org.membership_fee_paid,
                    'membership_expiry_date': org.membership_expiry_date,
                    'date_joined': org.date_joined,
                },
                'application': {
                    'application_id': application.application_id if application else None,
                    'application_status': application.application_status if application else None,
                    'submitted_date': application.submitted_date if application else None,
                    'reviewed_date': application.reviewed_date if application else None,
                    'organization_name': application.organization_name if application else None,
                    'invoicing_contact_email': application.invoicing_contact_email if application else None,
                    'invoicing_contact_phone': application.invoicing_contact_phone if application else None,
                } if application else None,
                'pending_payments': list(pending_payments),
                'status_info': status_info,
                'membership_fee': getattr(settings, 'MOMO_MEMBERSHIP_FEE', '200'),
                'currency': getattr(settings, 'MOMO_CURRENCY', 'EUR'),
            })
        else:
            # User exists but no organization - check if they have an application
            # This shouldn't normally happen, but handle it
            return Response({
                'success': True,
                'has_organization': False,
                'organization': None,
                'application': None,
                'pending_payments': [],
                'status_info': {
                    'status': 'NO_ORGANIZATION',
                    'status_label': 'No Organization',
                    'status_color': 'gray',
                    'message': 'Your account is not linked to any organization.',
                    'can_pay': False,
                    'next_steps': ['Please contact the NGO Forum secretariat for assistance.']
                },
                'membership_fee': getattr(settings, 'MOMO_MEMBERSHIP_FEE', '200'),
                'currency': getattr(settings, 'MOMO_CURRENCY', 'EUR'),
            })
    
    def _get_status_info(self, org, application):
        """Determine status info and next steps based on org and application status"""
        
        # If organization is active and paid
        if org.status == 'ACTIVE' and org.membership_fee_paid:
            return {
                'status': 'ACTIVE',
                'status_label': 'Active Member',
                'status_color': 'green',
                'message': 'Your membership is active. You have full access to all member features.',
                'can_pay': False,
                'next_steps': [
                    'Update your organization profile',
                    'Submit operational presence data',
                    'Participate in forum discussions',
                    'Access member resources'
                ]
            }
        
        # If organization is pending and application is pending payment
        if application and application.application_status == 'PENDING_PAYMENT':
            return {
                'status': 'PENDING_PAYMENT',
                'status_label': 'Pending Payment',
                'status_color': 'yellow',
                'message': 'Your application has been approved! Please pay the membership fee to activate your account.',
                'can_pay': True,
                'next_steps': [
                    'Pay the membership fee of $200 via MTN Mobile Money',
                    'Your account will be automatically activated after payment'
                ]
            }
        
        # If organization is pending (not paid yet)
        if org.status == 'PENDING':
            return {
                'status': 'PENDING',
                'status_label': 'Pending Activation',
                'status_color': 'yellow',
                'message': 'Your account is pending activation. Please complete payment or contact the secretariat.',
                'can_pay': True,
                'next_steps': [
                    'Pay the membership fee to activate your account',
                    'Contact the secretariat if you have questions'
                ]
            }
        
        # If organization is suspended
        if org.status == 'SUSPENDED':
            return {
                'status': 'SUSPENDED',
                'status_label': 'Suspended',
                'status_color': 'red',
                'message': 'Your membership has been suspended. Please contact the secretariat.',
                'can_pay': False,
                'next_steps': [
                    'Contact info@southsudanngoforum.org for more information'
                ]
            }
        
        # If organization is inactive
        if org.status == 'INACTIVE':
            return {
                'status': 'INACTIVE',
                'status_label': 'Inactive',
                'status_color': 'gray',
                'message': 'Your membership is inactive. Please renew your membership.',
                'can_pay': True,
                'next_steps': [
                    'Pay the membership fee to reactivate your account'
                ]
            }
        
        # Default case
        return {
            'status': org.status,
            'status_label': org.status,
            'status_color': 'gray',
            'message': 'Please contact the secretariat for more information.',
            'can_pay': False,
            'next_steps': ['Contact info@southsudanngoforum.org']
        }


class StaffMemberViewSet(viewsets.ReadOnlyModelViewSet):
    """Public access to staff directory"""
    queryset = StaffMember.objects.filter(is_active=True)
    serializer_class = StaffMemberSerializer
    permission_classes = [permissions.AllowAny]


class MembershipApplicationViewSet(viewsets.ModelViewSet):
    """Handle membership applications"""
    permission_classes = [permissions.AllowAny]
    parser_classes = [MultiPartParser, FormParser, JSONParser]
    
    def get_serializer_class(self):
        if self.action == 'create':
            return MembershipApplicationCreateSerializer
        return MembershipApplicationSerializer
    
    def get_queryset(self):
        if self.request.user.is_authenticated and self.request.user.is_staff:
            return MembershipApplication.objects.all()
        # Allow tracking by application_id
        application_id = self.request.query_params.get('application_id')
        if application_id:
            return MembershipApplication.objects.filter(application_id=application_id)
        return MembershipApplication.objects.none()
    
    def create(self, request, *args, **kwargs):
        serializer = self.get_serializer(data=request.data)
        serializer.is_valid(raise_exception=True)
        instance = serializer.save()
        
        # Return full application details with the application ID
        response_serializer = MembershipApplicationSerializer(instance)
        return Response({
            'success': True,
            'message': 'Your membership application has been submitted successfully.',
            'application_id': instance.application_id,
            'data': response_serializer.data
        }, status=status.HTTP_201_CREATED)
    
    @action(detail=False, methods=['get'], url_path='track/(?P<application_id>[^/.]+)')
    def track(self, request, application_id=None):
        """Track application status by application ID"""
        try:
            application = MembershipApplication.objects.get(application_id=application_id)
            serializer = MembershipApplicationSerializer(application)
            return Response({
                'success': True,
                'data': serializer.data
            })
        except MembershipApplication.DoesNotExist:
            return Response({
                'success': False,
                'message': 'Application not found'
            }, status=status.HTTP_404_NOT_FOUND)


class MembershipPaymentViewSet(viewsets.ModelViewSet):
    """Handle membership payments"""
    serializer_class = MembershipPaymentSerializer
    permission_classes = [permissions.IsAuthenticated]
    
    def get_queryset(self):
        if self.request.user.is_staff:
            return MembershipPayment.objects.all()
        if hasattr(self.request.user, 'member_organization'):
            return MembershipPayment.objects.filter(organization=self.request.user.member_organization)
        return MembershipPayment.objects.none()


# ========== MTN Mobile Money Payment Views ==========

@method_decorator(csrf_exempt, name='dispatch')
class MoMoCallbackView(APIView):
    """
    Webhook endpoint to receive MTN MoMo payment callbacks.
    
    MTN will POST to this endpoint when a payment request is processed.
    This will auto-approve membership applications upon successful payment.
    """
    permission_classes = [permissions.AllowAny]
    
    def post(self, request):
        """Handle incoming payment callback from MTN"""
        try:
            # Log the incoming callback for debugging
            print(f"MoMo Callback received: {request.data}")
            
            # Extract data from callback
            data = request.data
            external_id = data.get('externalId')
            status_value = data.get('status')  # SUCCESSFUL, FAILED, PENDING
            financial_transaction_id = data.get('financialTransactionId')
            reason = data.get('reason', '')
            
            if not external_id:
                return Response({'error': 'Missing externalId'}, status=status.HTTP_400_BAD_REQUEST)
            
            # Find the payment request
            try:
                payment = MoMoPaymentRequest.objects.get(external_id=external_id)
            except MoMoPaymentRequest.DoesNotExist:
                return Response({'error': 'Payment request not found'}, status=status.HTTP_404_NOT_FOUND)
            
            # Update payment status
            payment.status = status_value
            payment.status_reason = reason
            if financial_transaction_id:
                payment.financial_transaction_id = financial_transaction_id
            
            if status_value == 'SUCCESSFUL':
                payment.completed_at = timezone.now()
                # Auto-approve the application
                self._auto_approve_application(payment.application)
            
            payment.save()
            
            return Response({'success': True, 'message': 'Callback processed'})
            
        except Exception as e:
            print(f"MoMo Callback error: {str(e)}")
            return Response({'error': str(e)}, status=status.HTTP_500_INTERNAL_SERVER_ERROR)
    
    def _auto_approve_application(self, application):
        """Automatically approve application after successful payment"""
        if application.application_status == 'PENDING_PAYMENT':
            application.application_status = 'APPROVED'
            application.save()
            
            # Activate the member organization
            if application.approved_organization:
                org = application.approved_organization
                org.status = 'ACTIVE'
                org.membership_fee_paid = True
                # Set membership expiry to December 31st of the current year
                current_year = date.today().year
                org.membership_expiry_date = date(current_year, 12, 31)
                org.save()
                
                # Send approval confirmation email
                self._send_payment_confirmation_email(application)
    
    def _send_payment_confirmation_email(self, application):
        """Send confirmation email after payment is received"""
        subject = 'South Sudan NGO Forum - Payment Received - Membership Activated!'
        
        message = f"""Dear {application.invoicing_contact_name or application.organization_name},

Congratulations! We have received your membership payment for {application.organization_name}.

Your membership is now ACTIVE!

You can now login to the member portal with full access at:
https://ngo.intvault.online/portal/login

As a member, you now have access to:
- Member directory
- Forums and discussions
- Events and workshops
- Resources and documents
- Job postings
- And much more!

If you have any questions, please contact us at info@southsudanngoforum.org

Welcome to the South Sudan NGO Forum family!

Best regards,
South Sudan NGO Forum Secretariat
"""
        
        try:
            send_mail(
                subject=subject,
                message=message,
                from_email=settings.DEFAULT_FROM_EMAIL,
                recipient_list=[application.invoicing_contact_email],
                fail_silently=False,
            )
        except Exception as e:
            print(f"Payment confirmation email failed: {str(e)}")


class InitiatePaymentView(APIView):
    """
    Initiate a payment request for a membership application.
    
    This endpoint is called when a member wants to pay via MTN Mobile Money.
    It creates a payment request and sends it to the payer's phone.
    """
    permission_classes = [permissions.AllowAny]  # Allow public access for payment
    
    def post(self, request):
        """Initiate a payment request"""
        from .momo_service import get_momo_service
        
        application_id = request.data.get('application_id')
        phone_number = request.data.get('phone_number')
        
        if not application_id or not phone_number:
            return Response({
                'success': False,
                'error': 'application_id and phone_number are required'
            }, status=status.HTTP_400_BAD_REQUEST)
        
        # Find the application
        try:
            application = MembershipApplication.objects.get(application_id=application_id)
        except MembershipApplication.DoesNotExist:
            return Response({
                'success': False,
                'error': 'Application not found'
            }, status=status.HTTP_404_NOT_FOUND)
        
        # Check if application is pending payment
        if application.application_status != 'PENDING_PAYMENT':
            return Response({
                'success': False,
                'error': f'Application is not pending payment. Current status: {application.application_status}'
            }, status=status.HTTP_400_BAD_REQUEST)
        
        # Check for existing pending payment
        existing_payment = MoMoPaymentRequest.objects.filter(
            application=application,
            status='PENDING'
        ).first()
        
        if existing_payment:
            return Response({
                'success': False,
                'error': 'A payment request is already pending for this application',
                'external_id': existing_payment.external_id
            }, status=status.HTTP_400_BAD_REQUEST)
        
        # Format phone number (ensure no + sign, starts with country code)
        phone = phone_number.replace('+', '').replace(' ', '').replace('-', '')
        if phone.startswith('0'):
            phone = '256' + phone[1:]  # Assuming Uganda for MTN
        
        # Get membership fee and currency
        membership_fee = getattr(settings, 'MOMO_MEMBERSHIP_FEE', '200')
        currency = getattr(settings, 'MOMO_CURRENCY', 'EUR')  # EUR for sandbox, USD for production
        
        # Create external ID
        external_id = f"NGO-{application_id}-{uuid.uuid4().hex[:8].upper()}"
        
        # Create payment record
        payment = MoMoPaymentRequest.objects.create(
            application=application,
            amount=membership_fee,
            currency=currency,
            payer_phone=phone,
            external_id=external_id,
            payer_message=f'NGO Forum Membership - {application.organization_name}',
            payee_note=f'Membership fee for {application_id}'
        )
        
        # Initiate payment request with MTN
        momo = get_momo_service()
        result = momo.request_to_pay(
            amount=str(membership_fee),
            currency=currency,
            external_id=external_id,
            payer_phone=phone,
            payer_message=payment.payer_message,
            payee_note=payment.payee_note,
        )
        
        if result['success']:
            payment.reference_id = uuid.UUID(result['reference_id'])
            payment.save()
            
            return Response({
                'success': True,
                'message': 'Payment request sent. Please check your phone to approve the payment.',
                'external_id': external_id,
                'reference_id': result['reference_id'],
                'amount': membership_fee,
                'currency': currency
            })
        else:
            payment.status = 'FAILED'
            payment.status_reason = result.get('error', 'Unknown error')
            payment.save()
            
            return Response({
                'success': False,
                'error': result.get('error', 'Failed to initiate payment')
            }, status=status.HTTP_500_INTERNAL_SERVER_ERROR)


class CheckPaymentStatusView(APIView):
    """Check the status of a payment request"""
    permission_classes = [permissions.AllowAny]
    
    def get(self, request, external_id):
        """Get payment status by external ID"""
        from .momo_service import get_momo_service
        
        try:
            payment = MoMoPaymentRequest.objects.get(external_id=external_id)
        except MoMoPaymentRequest.DoesNotExist:
            return Response({
                'success': False,
                'error': 'Payment not found'
            }, status=status.HTTP_404_NOT_FOUND)
        
        # If still pending and has reference_id, check with MTN
        if payment.status == 'PENDING' and payment.reference_id:
            momo = get_momo_service()
            result = momo.get_payment_status(str(payment.reference_id))
            
            if result['success']:
                old_status = payment.status
                payment.status = result['status']
                
                if result.get('financial_transaction_id'):
                    payment.financial_transaction_id = result['financial_transaction_id']
                if result.get('reason'):
                    payment.status_reason = result['reason']
                
                if result['status'] == 'SUCCESSFUL' and old_status != 'SUCCESSFUL':
                    payment.completed_at = timezone.now()
                    # Auto-approve the application
                    self._auto_approve_application(payment.application)
                
                payment.save()
        
        return Response({
            'success': True,
            'external_id': payment.external_id,
            'status': payment.status,
            'amount': str(payment.amount),
            'currency': payment.currency,
            'financial_transaction_id': payment.financial_transaction_id,
            'application_status': payment.application.application_status,
            'completed_at': payment.completed_at.isoformat() if payment.completed_at else None
        })
    
    def _auto_approve_application(self, application):
        """Automatically approve application after successful payment"""
        if application.application_status == 'PENDING_PAYMENT':
            application.application_status = 'APPROVED'
            application.save()
            
            if application.approved_organization:
                org = application.approved_organization
                org.status = 'ACTIVE'
                org.membership_fee_paid = True
                # Set membership expiry to December 31st of the current year
                current_year = date.today().year
                org.membership_expiry_date = date(current_year, 12, 31)
                org.save()


@api_view(['GET'])
@perm_decorator([permissions.IsAdminUser])
def setup_momo_sandbox(request):
    """
    Setup MTN MoMo sandbox credentials.
    This creates an API user and API key for testing.
    Only accessible by admin users.
    """
    from .momo_service import get_momo_service
    
    momo = get_momo_service()
    result = momo.setup_sandbox()
    
    if result['success']:
        return Response({
            'success': True,
            'message': 'Sandbox credentials generated. Add these to your .env file:',
            'api_user_id': result['api_user_id'],
            'api_key': result['api_key'],
            'instructions': [
                f"MOMO_API_USER_ID={result['api_user_id']}",
                f"MOMO_API_KEY={result['api_key']}",
                'Then restart the Django server.'
            ]
        })
    else:
        return Response({
            'success': False,
            'error': result.get('error')
        }, status=status.HTTP_500_INTERNAL_SERVER_ERROR)
