from rest_framework import serializers
from .models import MemberOrganization, OrganizationContact, StaffMember, MembershipApplication, MembershipPayment


class OrganizationContactSerializer(serializers.ModelSerializer):
    class Meta:
        model = OrganizationContact
        fields = ['id', 'name', 'title', 'email', 'phone', 'is_primary', 'created_at']


class MemberOrganizationListSerializer(serializers.ModelSerializer):
    """Lightweight serializer for list views"""
    class Meta:
        model = MemberOrganization
        fields = [
            'id', 'name', 'slug', 'member_type', 'logo', 'city', 'state',
            'website', 'email', 'status', 'is_verified', 'date_joined'
        ]


class MemberOrganizationDetailSerializer(serializers.ModelSerializer):
    """Full serializer for detail views"""
    contacts = OrganizationContactSerializer(many=True, read_only=True)
    is_membership_expiring_soon = serializers.ReadOnlyField()
    
    class Meta:
        model = MemberOrganization
        fields = [
            'id', 'name', 'slug', 'member_type', 'rrc_number', 'registration_date',
            'email', 'phone', 'website', 'address', 'city', 'state', 'description',
            'logo', 'status', 'date_joined', 'is_verified', 'auto_approve_content',
            'membership_fee_paid', 'membership_expiry_date', 'is_membership_expiring_soon',
            'contacts', 'created_at', 'updated_at'
        ]
        read_only_fields = ['slug', 'is_verified', 'auto_approve_content', 'membership_fee_paid', 'membership_expiry_date']


class MemberOrganizationWriteSerializer(serializers.ModelSerializer):
    """Serializer for member profile updates"""
    class Meta:
        model = MemberOrganization
        fields = [
            'name', 'rrc_number', 'email', 'phone', 'website', 'address',
            'city', 'state', 'description', 'logo'
        ]


class StaffMemberSerializer(serializers.ModelSerializer):
    class Meta:
        model = StaffMember
        fields = ['id', 'name', 'position', 'email', 'phone', 'bio', 'photo', 'order', 'is_active']


class MembershipApplicationSerializer(serializers.ModelSerializer):
    """Full serializer for membership applications"""
    class Meta:
        model = MembershipApplication
        fields = [
            'id', 'application_id',
            # Step 1: Basic Info
            'organization_name', 'organization_acronym', 'organization_type',
            'organization_brief', 'is_non_political', 'is_non_proselytising',
            'is_humanitarian_actor', 'has_fulltime_presence', 'headquarter_country',
            'ss_main_office_state', 'ss_hq_physical_address', 'ngo_vehicle_plate_prefix',
            'year_operations_began', 'website',
            # Step 2: Other Info
            'rrc_certificate', 'rrc_registration_number', 'rrc_certificate_start_date',
            'rrc_certificate_expiry_date', 'is_woman_led', 'international_staff_count',
            'national_staff_count', 'relocatable_staff_count', 'juba_residence_compound',
            'annual_operating_budget_usd', 'invoicing_contact_name', 'invoicing_contact_email',
            'invoicing_contact_phone',
            # Step 3: Mailing List
            'mailing_list_contacts',
            # Step 4: Code of Conducts
            'psea_policy', 'codes_of_conduct', 'global_networks',
            # Step 5: Operational Presence
            'clusters_sectors', 'operational_presence',
            # Status
            'application_status', 'submitted_date', 'reviewed_date', 'reviewer_notes',
            'created_at', 'updated_at'
        ]
        read_only_fields = ['id', 'application_id', 'application_status', 'submitted_date', 'reviewed_date', 'reviewer_notes']


class MembershipApplicationCreateSerializer(serializers.ModelSerializer):
    """Serializer for creating/submitting membership applications"""
    class Meta:
        model = MembershipApplication
        fields = [
            # Step 1: Basic Info
            'organization_name', 'organization_acronym', 'organization_type',
            'organization_brief', 'is_non_political', 'is_non_proselytising',
            'is_humanitarian_actor', 'has_fulltime_presence', 'headquarter_country',
            'ss_main_office_state', 'ss_hq_physical_address', 'ngo_vehicle_plate_prefix',
            'year_operations_began', 'website',
            # Step 2: Other Info
            'rrc_certificate', 'rrc_registration_number', 'rrc_certificate_start_date',
            'rrc_certificate_expiry_date', 'is_woman_led', 'international_staff_count',
            'national_staff_count', 'relocatable_staff_count', 'juba_residence_compound',
            'annual_operating_budget_usd', 'invoicing_contact_name', 'invoicing_contact_email',
            'invoicing_contact_phone',
            # Step 3: Mailing List
            'mailing_list_contacts',
            # Step 4: Code of Conducts
            'psea_policy', 'codes_of_conduct', 'global_networks',
            # Step 5: Operational Presence
            'clusters_sectors', 'operational_presence',
        ]
    
    def validate(self, data):
        # Validate eligibility requirements
        errors = []
        if not data.get('is_non_political'):
            errors.append("NGO Forum Membership requires members to be Non Political")
        if not data.get('is_non_proselytising'):
            errors.append("NGO Forum Membership requires members to be Non Proselytising")
        if not data.get('is_humanitarian_actor'):
            errors.append("NGO Forum Membership requires members to be Humanitarian and Development actors")
        if not data.get('has_fulltime_presence'):
            errors.append("NGO Forum Membership requires members to have Fulltime Presence in South Sudan")
        
        if errors:
            raise serializers.ValidationError({"eligibility": errors})
        
        return data
    
    def create(self, validated_data):
        from django.utils import timezone
        validated_data['application_status'] = 'PENDING'
        validated_data['submitted_date'] = timezone.now()
        return super().create(validated_data)


class MembershipPaymentSerializer(serializers.ModelSerializer):
    organization_name = serializers.CharField(source='organization.name', read_only=True)
    
    class Meta:
        model = MembershipPayment
        fields = [
            'id', 'organization', 'organization_name', 'amount', 'payment_date',
            'transaction_reference', 'payment_method', 'status', 'receipt', 'notes'
        ]
        read_only_fields = ['payment_date', 'status']
