"""
MTN Mobile Money API Integration Service

This module handles the integration with MTN MoMo Collection API for membership payments.
Supports both sandbox (test) and production environments.

MTN MoMo API Documentation: https://momodeveloper.mtn.com/
"""

import requests
import uuid
import base64
from typing import Optional, Dict, Any
from django.conf import settings
from dataclasses import dataclass


@dataclass
class MoMoConfig:
    """MTN MoMo API Configuration"""
    primary_key: str
    secondary_key: str
    api_user_id: str = ""
    api_key: str = ""
    environment: str = "sandbox"  # 'sandbox' or 'production'
    callback_url: str = ""
    
    @property
    def base_url(self) -> str:
        if self.environment == "production":
            return "https://proxy.momoapi.mtn.com"
        return "https://sandbox.momodeveloper.mtn.com"
    
    @property
    def target_environment(self) -> str:
        return "sandbox" if self.environment == "sandbox" else "mtnuganda"


class MoMoService:
    """
    MTN Mobile Money Collection API Service
    
    Handles:
    - API User creation (sandbox)
    - Access token generation
    - Request to Pay (Collection)
    - Payment status checking
    """
    
    def __init__(self, config: Optional[MoMoConfig] = None):
        if config is None:
            config = self._get_config_from_settings()
        self.config = config
        self._access_token: Optional[str] = None
        self._token_expiry: Optional[float] = None
    
    def _get_config_from_settings(self) -> MoMoConfig:
        """Load configuration from Django settings"""
        return MoMoConfig(
            primary_key=getattr(settings, 'MOMO_PRIMARY_KEY', ''),
            secondary_key=getattr(settings, 'MOMO_SECONDARY_KEY', ''),
            api_user_id=getattr(settings, 'MOMO_API_USER_ID', ''),
            api_key=getattr(settings, 'MOMO_API_KEY', ''),
            environment=getattr(settings, 'MOMO_ENVIRONMENT', 'sandbox'),
            callback_url=getattr(settings, 'MOMO_CALLBACK_URL', ''),
        )
    
    def _get_headers(self, include_auth: bool = False) -> Dict[str, str]:
        """Build request headers"""
        headers = {
            "Content-Type": "application/json",
            "Ocp-Apim-Subscription-Key": self.config.primary_key,
        }
        
        if include_auth:
            if not self._access_token:
                self._access_token = self.get_access_token()
            headers["Authorization"] = f"Bearer {self._access_token}"
        
        return headers
    
    # ========== Sandbox Setup Methods ==========
    
    def create_api_user(self) -> Dict[str, Any]:
        """
        Create an API User in sandbox environment.
        This is only needed for sandbox - in production, you get these from MTN.
        
        Returns:
            Dict with 'success', 'api_user_id', and optional 'error'
        """
        if self.config.environment != "sandbox":
            return {"success": False, "error": "API user creation is only for sandbox"}
        
        api_user_id = str(uuid.uuid4())
        
        url = f"{self.config.base_url}/v1_0/apiuser"
        headers = {
            "Content-Type": "application/json",
            "X-Reference-Id": api_user_id,
            "Ocp-Apim-Subscription-Key": self.config.primary_key,
        }
        
        # providerCallbackHost is the domain that will receive callbacks
        # For sandbox, this just needs to be a valid URL format
        callback_host = self.config.callback_url or "http://localhost:8000"
        # Extract just the host from the URL if it's a full URL
        if callback_host.startswith("http"):
            from urllib.parse import urlparse
            parsed = urlparse(callback_host)
            callback_host = f"{parsed.scheme}://{parsed.netloc}"
        
        payload = {
            "providerCallbackHost": callback_host
        }
        
        try:
            response = requests.post(url, json=payload, headers=headers, timeout=30)
            
            if response.status_code == 201:
                return {"success": True, "api_user_id": api_user_id}
            else:
                return {
                    "success": False,
                    "error": f"Failed to create API user: {response.status_code} - {response.text}"
                }
        except requests.RequestException as e:
            return {"success": False, "error": str(e)}
    
    def create_api_key(self, api_user_id: str) -> Dict[str, Any]:
        """
        Create an API Key for the given API User (sandbox only).
        
        Args:
            api_user_id: The UUID of the API user
            
        Returns:
            Dict with 'success', 'api_key', and optional 'error'
        """
        if self.config.environment != "sandbox":
            return {"success": False, "error": "API key creation is only for sandbox"}
        
        url = f"{self.config.base_url}/v1_0/apiuser/{api_user_id}/apikey"
        headers = {
            "Ocp-Apim-Subscription-Key": self.config.primary_key,
        }
        
        try:
            response = requests.post(url, headers=headers, timeout=30)
            
            if response.status_code == 201:
                data = response.json()
                return {"success": True, "api_key": data.get("apiKey")}
            else:
                return {
                    "success": False,
                    "error": f"Failed to create API key: {response.status_code} - {response.text}"
                }
        except requests.RequestException as e:
            return {"success": False, "error": str(e)}
    
    def setup_sandbox(self) -> Dict[str, Any]:
        """
        Complete sandbox setup: create API user and API key.
        
        Returns:
            Dict with 'success', 'api_user_id', 'api_key', and optional 'error'
        """
        # Create API user
        user_result = self.create_api_user()
        if not user_result["success"]:
            return user_result
        
        api_user_id = user_result["api_user_id"]
        
        # Create API key
        key_result = self.create_api_key(api_user_id)
        if not key_result["success"]:
            return key_result
        
        return {
            "success": True,
            "api_user_id": api_user_id,
            "api_key": key_result["api_key"],
            "message": "Add these to your .env file as MOMO_API_USER_ID and MOMO_API_KEY"
        }
    
    # ========== Authentication ==========
    
    def get_access_token(self) -> Optional[str]:
        """
        Get an OAuth access token for API calls.
        
        Returns:
            Access token string or None if failed
        """
        url = f"{self.config.base_url}/collection/token/"
        
        # Basic auth with API user ID and API key
        credentials = f"{self.config.api_user_id}:{self.config.api_key}"
        encoded_credentials = base64.b64encode(credentials.encode()).decode()
        
        headers = {
            "Authorization": f"Basic {encoded_credentials}",
            "Ocp-Apim-Subscription-Key": self.config.primary_key,
        }
        
        try:
            response = requests.post(url, headers=headers, timeout=30)
            
            if response.status_code == 200:
                data = response.json()
                self._access_token = data.get("access_token")
                return self._access_token
            else:
                print(f"Failed to get access token: {response.status_code} - {response.text}")
                return None
        except requests.RequestException as e:
            print(f"Token request failed: {str(e)}")
            return None
    
    # ========== Collection (Request to Pay) ==========
    
    def request_to_pay(
        self,
        amount: str,
        currency: str,
        external_id: str,
        payer_phone: str,
        payer_message: str = "",
        payee_note: str = "",
    ) -> Dict[str, Any]:
        """
        Initiate a Request to Pay (collection) from a mobile money user.
        
        Args:
            amount: Amount to collect (as string, e.g., "200")
            currency: Currency code (e.g., "USD", "EUR", "UGX")
            external_id: Your unique reference for this transaction
            payer_phone: Phone number in format 256xxxxxxxxx (no + sign)
            payer_message: Message to display to payer
            payee_note: Note for your records
            
        Returns:
            Dict with 'success', 'reference_id', and optional 'error'
        """
        url = f"{self.config.base_url}/collection/v1_0/requesttopay"
        
        reference_id = str(uuid.uuid4())
        
        headers = self._get_headers(include_auth=True)
        headers["X-Reference-Id"] = reference_id
        headers["X-Target-Environment"] = self.config.target_environment
        
        # Note: X-Callback-Url is optional and must match the providerCallbackHost
        # configured when creating the API user. For sandbox testing, we skip it
        # since callbacks won't work to localhost anyway.
        # In production, ensure MOMO_CALLBACK_URL matches the configured host.
        
        payload = {
            "amount": amount,
            "currency": currency,
            "externalId": external_id,
            "payer": {
                "partyIdType": "MSISDN",
                "partyId": payer_phone
            },
            "payerMessage": payer_message or f"NGO Forum Membership Fee - {external_id}",
            "payeeNote": payee_note or f"Membership payment {external_id}"
        }
        
        try:
            response = requests.post(url, json=payload, headers=headers, timeout=30)
            
            if response.status_code == 202:
                return {
                    "success": True,
                    "reference_id": reference_id,
                    "message": "Payment request sent. Waiting for user approval."
                }
            else:
                return {
                    "success": False,
                    "error": f"Request to pay failed: {response.status_code} - {response.text}"
                }
        except requests.RequestException as e:
            return {"success": False, "error": str(e)}
    
    def get_payment_status(self, reference_id: str) -> Dict[str, Any]:
        """
        Check the status of a payment request.
        
        Args:
            reference_id: The X-Reference-Id from request_to_pay
            
        Returns:
            Dict with payment status details
        """
        url = f"{self.config.base_url}/collection/v1_0/requesttopay/{reference_id}"
        
        headers = self._get_headers(include_auth=True)
        headers["X-Target-Environment"] = self.config.target_environment
        
        try:
            response = requests.get(url, headers=headers, timeout=30)
            
            if response.status_code == 200:
                data = response.json()
                return {
                    "success": True,
                    "status": data.get("status"),  # PENDING, SUCCESSFUL, FAILED
                    "external_id": data.get("externalId"),
                    "amount": data.get("amount"),
                    "currency": data.get("currency"),
                    "financial_transaction_id": data.get("financialTransactionId"),
                    "payer": data.get("payer", {}),
                    "reason": data.get("reason", ""),
                    "raw": data
                }
            else:
                return {
                    "success": False,
                    "error": f"Status check failed: {response.status_code} - {response.text}"
                }
        except requests.RequestException as e:
            return {"success": False, "error": str(e)}
    
    def get_account_balance(self) -> Dict[str, Any]:
        """
        Get the balance of the collection account.
        
        Returns:
            Dict with balance details
        """
        url = f"{self.config.base_url}/collection/v1_0/account/balance"
        
        headers = self._get_headers(include_auth=True)
        headers["X-Target-Environment"] = self.config.target_environment
        
        try:
            response = requests.get(url, headers=headers, timeout=30)
            
            if response.status_code == 200:
                data = response.json()
                return {
                    "success": True,
                    "available_balance": data.get("availableBalance"),
                    "currency": data.get("currency")
                }
            else:
                return {
                    "success": False,
                    "error": f"Balance check failed: {response.status_code} - {response.text}"
                }
        except requests.RequestException as e:
            return {"success": False, "error": str(e)}


# Singleton instance for easy import
def get_momo_service() -> MoMoService:
    """Get the MoMo service instance"""
    return MoMoService()
