from django.db import models
from django.contrib.auth.models import User
from django.core.validators import FileExtensionValidator, MaxValueValidator
from django.core.exceptions import ValidationError
from django.utils.text import slugify
import os


def validate_file_size(value):
    """Validate that file size is under 15MB"""
    filesize = value.size
    if filesize > 15 * 1024 * 1024:  # 15MB
        raise ValidationError("Maximum file size is 15MB")
    return value


class TimeStampedModel(models.Model):
    """Abstract base model with timestamps"""
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)
    
    class Meta:
        abstract = True


class MemberOrganization(TimeStampedModel):
    """NGO Member Organizations"""
    MEMBER_TYPE_CHOICES = [
        ('NATIONAL', 'National NGO'),
        ('INTERNATIONAL', 'International NGO'),
    ]
    
    STATUS_CHOICES = [
        ('ACTIVE', 'Active'),
        ('PENDING', 'Pending'),
        ('INACTIVE', 'Inactive'),
        ('SUSPENDED', 'Suspended'),
    ]
    
    name = models.CharField(max_length=255)
    slug = models.SlugField(max_length=255, unique=True, blank=True)
    member_type = models.CharField(max_length=20, choices=MEMBER_TYPE_CHOICES)
    rrc_number = models.CharField(max_length=100, blank=True, help_text="RRC Registration Number")
    registration_date = models.DateField(null=True, blank=True)
    
    # Contact information
    email = models.EmailField()
    phone = models.CharField(max_length=50, blank=True)
    website = models.URLField(blank=True)
    address = models.TextField()
    city = models.CharField(max_length=100, default='Juba')
    state = models.CharField(max_length=100, blank=True)
    
    # Organization details
    description = models.TextField(blank=True)
    logo = models.ImageField(upload_to='logos/', blank=True, validators=[validate_file_size])
    
    # Membership status
    status = models.CharField(max_length=20, choices=STATUS_CHOICES, default='PENDING')
    date_joined = models.DateField(auto_now_add=True)
    
    # Verification and auto-approval
    is_verified = models.BooleanField(default=False, help_text="Verified members get auto-approval for content")
    auto_approve_content = models.BooleanField(default=False)
    
    # Payment tracking
    membership_fee_paid = models.BooleanField(default=False)
    membership_expiry_date = models.DateField(null=True, blank=True)
    
    # User association
    user = models.OneToOneField(User, on_delete=models.SET_NULL, null=True, blank=True, related_name='member_organization')
    
    class Meta:
        ordering = ['name']
        verbose_name = 'Member Organization'
        verbose_name_plural = 'Member Organizations'
    
    def __str__(self):
        return self.name
    
    def save(self, *args, **kwargs):
        if not self.slug:
            self.slug = slugify(self.name)
        super().save(*args, **kwargs)
    
    @property
    def is_membership_expiring_soon(self):
        """Check if membership expires within 30 days"""
        if not self.membership_expiry_date:
            return False
        from datetime import date, timedelta
        return date.today() + timedelta(days=30) >= self.membership_expiry_date >= date.today()


class OrganizationContact(TimeStampedModel):
    """Contact persons for member organizations"""
    organization = models.ForeignKey(MemberOrganization, on_delete=models.CASCADE, related_name='contacts')
    name = models.CharField(max_length=200)
    title = models.CharField(max_length=200)
    email = models.EmailField()
    phone = models.CharField(max_length=50)
    is_primary = models.BooleanField(default=False)
    
    class Meta:
        ordering = ['-is_primary', 'name']
        verbose_name = 'Organization Contact'
        verbose_name_plural = 'Organization Contacts'
    
    def __str__(self):
        return f"{self.name} - {self.organization.name}"


class StaffMember(TimeStampedModel):
    """NGO Forum Secretariat Staff"""
    name = models.CharField(max_length=200)
    position = models.CharField(max_length=200)
    email = models.EmailField()
    phone = models.CharField(max_length=50, blank=True)
    bio = models.TextField(blank=True)
    photo = models.ImageField(upload_to='staff/', blank=True, validators=[validate_file_size])
    order = models.IntegerField(default=0, help_text="Display order (lower numbers first)")
    is_active = models.BooleanField(default=True)
    
    class Meta:
        ordering = ['order', 'name']
        verbose_name = 'Staff Member'
        verbose_name_plural = 'Staff Members'
    
    def __str__(self):
        return f"{self.name} - {self.position}"


class MembershipApplication(TimeStampedModel):
    """New membership applications - comprehensive form matching the services portal"""
    ORG_TYPE_CHOICES = [
        ('INGO', 'International NGO'),
        ('NNGO', 'National NGO'),
        ('OBSERVER', 'Observer'),
    ]
    
    STATUS_CHOICES = [
        ('DRAFT', 'Draft'),
        ('PENDING', 'Pending Review'),
        ('STEERING_COMMITTEE', 'Steering Committee Review'),
        ('COUNTRY_DIRECTORS', 'Country Directors Vote'),
        ('PENDING_PAYMENT', 'Pending Payment'),
        ('APPROVED', 'Approved'),
        ('REJECTED', 'Rejected'),
    ]
    
    PSEA_CHOICES = [
        ('YES', 'Yes'),
        ('NO', 'No'),
        ('WORKING', 'We are working on it'),
        ('UNKNOWN', "I Don't Know"),
    ]
    
    WOMAN_LED_CHOICES = [
        ('YES', 'Yes'),
        ('NO', 'No'),
    ]
    
    # Application ID - auto generated
    application_id = models.CharField(max_length=20, unique=True, blank=True)
    
    # ===== STEP 1: Organization Basic Information =====
    organization_name = models.CharField(max_length=255)
    organization_acronym = models.CharField(max_length=50, default='')
    organization_type = models.CharField(max_length=20, choices=ORG_TYPE_CHOICES, default='NNGO')
    organization_brief = models.TextField(help_text="Short brief of your organization", default='')
    
    # Eligibility questions (must be YES for membership)
    is_non_political = models.BooleanField(default=False)
    is_non_proselytising = models.BooleanField(default=False)
    is_humanitarian_actor = models.BooleanField(default=False)
    has_fulltime_presence = models.BooleanField(default=False)
    
    # Location info
    headquarter_country = models.CharField(max_length=100, default='South Sudan')
    ss_main_office_state = models.CharField(max_length=100, default='Central Equatoria')
    ss_hq_physical_address = models.TextField(default='')
    ngo_vehicle_plate_prefix = models.CharField(max_length=20, blank=True, default='')
    year_operations_began = models.IntegerField(default=2020)
    website = models.URLField(blank=True, default='')
    
    # ===== STEP 2: Organization Other Information =====
    rrc_certificate = models.ImageField(
        upload_to='applications/certificates/',
        validators=[FileExtensionValidator(['jpg', 'jpeg', 'png']), validate_file_size],
        help_text="Upload RRC Certificate (2MB max, jpeg/png)",
        blank=True, null=True
    )
    rrc_registration_number = models.CharField(max_length=100, default='')
    rrc_certificate_start_date = models.DateField(null=True, blank=True)
    rrc_certificate_expiry_date = models.DateField(null=True, blank=True)
    
    is_woman_led = models.CharField(max_length=10, choices=WOMAN_LED_CHOICES, default='NO')
    
    # Staffing numbers
    international_staff_count = models.IntegerField(default=0)
    national_staff_count = models.IntegerField(default=0)
    relocatable_staff_count = models.IntegerField(default=0)
    
    juba_residence_compound = models.CharField(max_length=255, blank=True, default='')
    annual_operating_budget_usd = models.DecimalField(max_digits=15, decimal_places=2, null=True, blank=True)
    
    # Invoicing contact for membership fee
    invoicing_contact_name = models.CharField(max_length=200, default='')
    invoicing_contact_email = models.EmailField(default='')
    invoicing_contact_phone = models.CharField(max_length=50, default='')
    
    # ===== STEP 3: Mailing List Information (stored as JSON) =====
    mailing_list_contacts = models.JSONField(default=list, blank=True, help_text="List of contacts for mailing lists")
    
    # ===== STEP 4: Code of Conducts =====
    psea_policy = models.CharField(max_length=20, choices=PSEA_CHOICES, default='UNKNOWN')
    codes_of_conduct = models.JSONField(default=list, blank=True, help_text="List of codes organization is signatory to")
    global_networks = models.JSONField(default=list, blank=True, help_text="List of global networks organization is part of")
    
    # ===== STEP 5: Operational Presence =====
    # Clusters/Sectors selected
    clusters_sectors = models.JSONField(default=list, help_text="List of clusters/sectors")
    # County presence per cluster - dict mapping cluster to list of counties
    operational_presence = models.JSONField(default=dict, help_text="Dict mapping cluster to list of counties")
    
    # ===== Application Status & Tracking =====
    application_status = models.CharField(max_length=30, choices=STATUS_CHOICES, default='DRAFT')
    submitted_date = models.DateTimeField(null=True, blank=True)
    reviewed_date = models.DateTimeField(null=True, blank=True)
    reviewer = models.ForeignKey(User, on_delete=models.SET_NULL, null=True, blank=True, related_name='reviewed_applications')
    reviewer_notes = models.TextField(blank=True)
    
    # Link to created organization if approved
    approved_organization = models.OneToOneField(
        MemberOrganization,
        on_delete=models.SET_NULL,
        null=True,
        blank=True,
        related_name='application'
    )
    
    class Meta:
        ordering = ['-submitted_date', '-created_at']
        verbose_name = 'Membership Application'
        verbose_name_plural = 'Membership Applications'
    
    def __str__(self):
        return f"{self.application_id} - {self.organization_name} ({self.application_status})"
    
    def save(self, *args, **kwargs):
        if not self.application_id:
            import random
            import string
            from datetime import datetime
            # Generate application ID: SSNGOF-YYYYMM-XXXXX
            year_month = datetime.now().strftime('%Y%m')
            random_chars = ''.join(random.choices(string.ascii_uppercase + string.digits, k=5))
            self.application_id = f"SSNGOF-{year_month}-{random_chars}"
        super().save(*args, **kwargs)


class MembershipPayment(TimeStampedModel):
    """Membership fee payments"""
    STATUS_CHOICES = [
        ('PENDING', 'Pending'),
        ('COMPLETED', 'Completed'),
        ('FAILED', 'Failed'),
        ('REFUNDED', 'Refunded'),
    ]
    
    organization = models.ForeignKey(MemberOrganization, on_delete=models.CASCADE, related_name='payments')
    amount = models.DecimalField(max_digits=10, decimal_places=2, default=200.00)
    payment_date = models.DateField(auto_now_add=True)
    transaction_reference = models.CharField(max_length=255, blank=True)
    payment_method = models.CharField(max_length=100, blank=True)
    status = models.CharField(max_length=20, choices=STATUS_CHOICES, default='PENDING')
    receipt = models.FileField(
        upload_to='payments/receipts/',
        blank=True,
        validators=[FileExtensionValidator(['pdf', 'jpg', 'jpeg', 'png']), validate_file_size]
    )
    notes = models.TextField(blank=True)
    
    class Meta:
        ordering = ['-payment_date']
        verbose_name = 'Membership Payment'
        verbose_name_plural = 'Membership Payments'
    
    def __str__(self):
        return f"{self.organization.name} - ${self.amount} ({self.status})"


class MoMoPaymentRequest(TimeStampedModel):
    """Track MTN Mobile Money payment requests for membership applications"""
    STATUS_CHOICES = [
        ('PENDING', 'Pending'),
        ('SUCCESSFUL', 'Successful'),
        ('FAILED', 'Failed'),
        ('CANCELLED', 'Cancelled'),
        ('TIMEOUT', 'Timeout'),
    ]
    
    # Link to application
    application = models.ForeignKey(
        MembershipApplication,
        on_delete=models.CASCADE,
        related_name='momo_payments'
    )
    
    # Payment details
    amount = models.DecimalField(max_digits=10, decimal_places=2, default=200.00)
    currency = models.CharField(max_length=3, default='USD')
    payer_phone = models.CharField(max_length=20, help_text="Phone number in format 256xxxxxxxxx")
    payer_message = models.TextField(blank=True)
    payee_note = models.TextField(blank=True)
    
    # MTN MoMo reference IDs
    external_id = models.CharField(max_length=100, unique=True, help_text="Our reference ID")
    reference_id = models.UUIDField(null=True, blank=True, help_text="MTN API request reference")
    financial_transaction_id = models.CharField(max_length=100, blank=True, help_text="MTN financial transaction ID")
    
    # Status tracking
    status = models.CharField(max_length=20, choices=STATUS_CHOICES, default='PENDING')
    status_reason = models.TextField(blank=True)
    
    # Timestamps
    requested_at = models.DateTimeField(auto_now_add=True)
    completed_at = models.DateTimeField(null=True, blank=True)
    
    class Meta:
        ordering = ['-created_at']
        verbose_name = 'MoMo Payment Request'
        verbose_name_plural = 'MoMo Payment Requests'
    
    def __str__(self):
        return f"{self.external_id} - {self.application.organization_name} - {self.status}"
