from django.contrib import admin
from django.contrib.auth.models import User
from django.core.mail import send_mail
from django.conf import settings
from django.contrib import messages
from django.utils import timezone
import string
import secrets
from .models import MemberOrganization, OrganizationContact, StaffMember, MembershipApplication, MembershipPayment, MoMoPaymentRequest


class OrganizationContactInline(admin.TabularInline):
    model = OrganizationContact
    extra = 1


@admin.register(MemberOrganization)
class MemberOrganizationAdmin(admin.ModelAdmin):
    list_display = ['name', 'member_type', 'is_verified', 'membership_fee_paid', 'membership_expiry_date', 'status', 'date_joined']
    list_filter = ['member_type', 'is_verified', 'status', 'state', 'membership_fee_paid']
    search_fields = ['name', 'email', 'rrc_number', 'description']
    readonly_fields = ['slug', 'date_joined', 'created_at', 'updated_at']
    date_hierarchy = 'date_joined'
    inlines = [OrganizationContactInline]
    
    fieldsets = (
        ('Basic Information', {
            'fields': ('name', 'slug', 'member_type', 'rrc_number', 'registration_date', 'logo')
        }),
        ('Contact Information', {
            'fields': ('email', 'phone', 'website', 'address', 'city', 'state')
        }),
        ('Description', {
            'fields': ('description',)
        }),
        ('Membership Status', {
            'fields': ('status', 'date_joined', 'is_verified', 'auto_approve_content')
        }),
        ('Payment & Expiry', {
            'fields': ('membership_fee_paid', 'membership_expiry_date')
        }),
        ('System', {
            'fields': ('user', 'created_at', 'updated_at'),
            'classes': ('collapse',)
        }),
    )
    
    actions = ['mark_verified', 'mark_unverified']
    
    def mark_verified(self, request, queryset):
        queryset.update(is_verified=True, auto_approve_content=True)
        self.message_user(request, f"{queryset.count()} members marked as verified")
    mark_verified.short_description = "Mark selected as verified"
    
    def mark_unverified(self, request, queryset):
        queryset.update(is_verified=False, auto_approve_content=False)
        self.message_user(request, f"{queryset.count()} members marked as unverified")
    mark_unverified.short_description = "Mark selected as unverified"


@admin.register(OrganizationContact)
class OrganizationContactAdmin(admin.ModelAdmin):
    list_display = ['name', 'organization', 'title', 'email', 'phone', 'is_primary']
    list_filter = ['is_primary']
    search_fields = ['name', 'email', 'organization__name']


@admin.register(StaffMember)
class StaffMemberAdmin(admin.ModelAdmin):
    list_display = ['name', 'position', 'email', 'order', 'is_active']
    list_filter = ['is_active']
    search_fields = ['name', 'position', 'email']
    ordering = ['order', 'name']


def generate_secure_password(length=12):
    """Generate a secure random password"""
    alphabet = string.ascii_letters + string.digits + string.punctuation.replace('"', '').replace("'", '').replace('\\', '')
    password = ''.join(secrets.choice(alphabet) for i in range(length))
    return password


@admin.register(MembershipApplication)
class MembershipApplicationAdmin(admin.ModelAdmin):
    list_display = ['application_id', 'organization_name', 'organization_type', 'application_status', 'submitted_date']
    list_filter = ['organization_type', 'application_status', 'is_woman_led', 'psea_policy', 'submitted_date']
    search_fields = ['application_id', 'organization_name', 'organization_acronym', 'rrc_registration_number', 'invoicing_contact_email']
    readonly_fields = ['application_id', 'submitted_date', 'reviewed_date', 'created_at', 'updated_at']
    date_hierarchy = 'submitted_date'
    
    def save_model(self, request, obj, form, change):
        """Override save to handle approval workflow"""
        # Check if status changed
        if change and 'application_status' in form.changed_data:
            # Handle PENDING_PAYMENT status - create account and send payment instructions
            if obj.application_status == 'PENDING_PAYMENT':
                try:
                    self._handle_pending_payment(request, obj)
                except Exception as e:
                    messages.error(request, f"Error processing pending payment: {str(e)}")
            
            # Handle direct APPROVED status (legacy or manual approval after payment)
            elif obj.application_status == 'APPROVED':
                # Only create account if not already created
                if not obj.approved_organization:
                    try:
                        self._create_member_account(request, obj, send_payment_email=False)
                    except Exception as e:
                        messages.error(request, f"Error creating member account: {str(e)}")
        
        super().save_model(request, obj, form, change)
    
    def _handle_pending_payment(self, request, application):
        """Create user account and send payment instructions email"""
        # Generate username from application_id
        username = application.application_id.replace('-', '').lower()
        
        # Check if user already exists
        if User.objects.filter(username=username).exists():
            messages.warning(request, f"User account '{username}' already exists for this application.")
            # Still send payment reminder email
            self._send_payment_reminder_email(application)
            return
        
        # Generate secure random password
        password = generate_secure_password(12)
        
        # Create the user
        user = User.objects.create_user(
            username=username,
            email=application.invoicing_contact_email,
            password=password,
            first_name=application.organization_name[:30],
        )
        user.save()
        
        # Create MemberOrganization linked to this user (with PENDING status)
        member_type = 'INTERNATIONAL' if application.organization_type == 'INGO' else 'NATIONAL'
        
        member_org = MemberOrganization.objects.create(
            name=application.organization_name,
            member_type=member_type,
            rrc_number=application.rrc_registration_number,
            email=application.invoicing_contact_email,
            phone=application.invoicing_contact_phone,
            website=application.website,
            address=application.ss_hq_physical_address,
            city='Juba',
            state=application.ss_main_office_state,
            description=application.organization_brief,
            status='PENDING',  # Will be set to ACTIVE after payment
            membership_fee_paid=False,
            user=user,
        )
        
        # Link approved organization to application
        application.approved_organization = member_org
        application.reviewed_date = timezone.now()
        application.reviewer = request.user
        
        # Send email with login credentials AND payment instructions
        email_sent = self._send_payment_pending_email(application, username, password)
        
        if email_sent:
            messages.success(
                request, 
                f"Member account created with PENDING status. Username: {username}. "
                f"Payment instructions sent to {application.invoicing_contact_email}"
            )
        else:
            messages.success(
                request, 
                f"Member account created! Username: {username}, Password: {password}. "
                f"(Email could not be sent - please share credentials and payment info manually)"
            )
    
    def _send_payment_pending_email(self, application, username, password):
        """Send email with login credentials and payment instructions"""
        membership_fee = getattr(settings, 'MOMO_MEMBERSHIP_FEE', '200')
        
        subject = 'South Sudan NGO Forum - Application Reviewed - Payment Required'
        
        message = f"""Dear {application.invoicing_contact_name or application.organization_name},

Great news! Your membership application for {application.organization_name} has been reviewed and approved!

To complete your membership, please pay the membership fee of ${membership_fee} USD.

=== YOUR ACCOUNT CREDENTIALS ===

Your member portal login credentials are:

    Username: {username}
    Password: {password}

You can login at: https://ngo.intvault.online/portal/login

Please change your password after your first login for security purposes.

=== PAYMENT INSTRUCTIONS ===

OPTION 1: MTN Mobile Money
- You will receive a payment request on your registered phone number
- Amount: ${membership_fee} USD
- Simply approve the payment when prompted

OPTION 2: Bank Transfer
- Bank: [Bank Name]
- Account Name: South Sudan NGO Forum
- Account Number: [Account Number]
- Reference: {application.application_id}

=== IMPORTANT ===

Your membership will be automatically activated once payment is confirmed.
Your account currently has limited access until payment is completed.

If you have any questions, please contact us at info@southsudanngoforum.org

Best regards,
South Sudan NGO Forum Secretariat
"""
        
        try:
            send_mail(
                subject=subject,
                message=message,
                from_email=settings.DEFAULT_FROM_EMAIL,
                recipient_list=[application.invoicing_contact_email],
                fail_silently=False,
            )
            return True
        except Exception as e:
            print(f"Email sending failed: {str(e)}")
            return False
    
    def _send_payment_reminder_email(self, application):
        """Send payment reminder email for existing account"""
        membership_fee = getattr(settings, 'MOMO_MEMBERSHIP_FEE', '200')
        
        subject = 'South Sudan NGO Forum - Payment Reminder'
        
        message = f"""Dear {application.invoicing_contact_name or application.organization_name},

This is a reminder that your membership application for {application.organization_name} is pending payment.

To complete your membership, please pay the membership fee of ${membership_fee} USD.

=== PAYMENT INSTRUCTIONS ===

OPTION 1: MTN Mobile Money
- You will receive a payment request on your registered phone number
- Amount: ${membership_fee} USD
- Simply approve the payment when prompted

OPTION 2: Bank Transfer
- Bank: [Bank Name]
- Account Name: South Sudan NGO Forum
- Account Number: [Account Number]
- Reference: {application.application_id}

Your membership will be automatically activated once payment is confirmed.

If you have any questions, please contact us at info@southsudanngoforum.org

Best regards,
South Sudan NGO Forum Secretariat
"""
        
        try:
            send_mail(
                subject=subject,
                message=message,
                from_email=settings.DEFAULT_FROM_EMAIL,
                recipient_list=[application.invoicing_contact_email],
                fail_silently=False,
            )
            return True
        except Exception as e:
            print(f"Email sending failed: {str(e)}")
            return False
        
        super().save_model(request, obj, form, change)
    
    def _create_member_account(self, request, application, send_payment_email=False):
        """Create a user account for the approved member and send email"""
        # Generate username from application_id (remove dashes and lowercase)
        username = application.application_id.replace('-', '').lower()
        
        # Check if user already exists
        if User.objects.filter(username=username).exists():
            messages.warning(request, f"User account '{username}' already exists for this application.")
            # If org exists, just activate it
            if application.approved_organization:
                application.approved_organization.status = 'ACTIVE'
                application.approved_organization.membership_fee_paid = True
                application.approved_organization.save()
                messages.success(request, f"Organization '{application.organization_name}' has been activated.")
            return
        
        # Generate secure random password
        password = generate_secure_password(12)
        
        # Create the user
        user = User.objects.create_user(
            username=username,
            email=application.invoicing_contact_email,
            password=password,
            first_name=application.organization_name[:30],  # User model first_name is limited
        )
        user.save()
        
        # Create MemberOrganization linked to this user
        member_type = 'INTERNATIONAL' if application.organization_type == 'INGO' else 'NATIONAL'
        
        member_org = MemberOrganization.objects.create(
            name=application.organization_name,
            member_type=member_type,
            rrc_number=application.rrc_registration_number,
            email=application.invoicing_contact_email,
            phone=application.invoicing_contact_phone,
            website=application.website,
            address=application.ss_hq_physical_address,
            city='Juba',
            state=application.ss_main_office_state,
            description=application.organization_brief,
            status='ACTIVE',
            membership_fee_paid=True,
            user=user,
        )
        
        # Link approved organization to application
        application.approved_organization = member_org
        application.reviewed_date = timezone.now()
        application.reviewer = request.user
        
        # Send email with login credentials (don't fail if email fails)
        email_sent = self._send_approval_email(application, username, password)
        
        if email_sent:
            messages.success(
                request, 
                f"Member account created! Username: {username}. "
                f"Login details sent to {application.invoicing_contact_email}"
            )
        else:
            messages.success(
                request, 
                f"Member account created! Username: {username}, Password: {password}. "
                f"(Email could not be sent - please share credentials manually)"
            )
    
    def _send_approval_email(self, application, username, password):
        """Send approval email with login credentials"""
        subject = 'South Sudan NGO Forum - Membership Approved!'
        
        message = f"""Dear {application.invoicing_contact_name or application.organization_name},

Congratulations! Your membership application for {application.organization_name} has been approved.

Your member portal login credentials are:

    Username: {username}
    Password: {password}

You can login at: https://ngo.intvault.online/portal/login

Please change your password after your first login for security purposes.

If you have any questions, please contact us at info@southsudanngoforum.org

Best regards,
South Sudan NGO Forum Secretariat
"""
        
        try:
            send_mail(
                subject=subject,
                message=message,
                from_email=settings.DEFAULT_FROM_EMAIL,
                recipient_list=[application.invoicing_contact_email],
                fail_silently=False,
            )
            return True
        except Exception as e:
            # Log the error but don't break the approval process
            print(f"Email sending failed: {str(e)}")
            return False
    
    fieldsets = (
        ('Application ID', {
            'fields': ('application_id', 'application_status')
        }),
        ('Step 1: Organization Basic Information', {
            'fields': (
                'organization_name', 'organization_acronym', 'organization_type',
                'organization_brief', 'is_non_political', 'is_non_proselytising',
                'is_humanitarian_actor', 'has_fulltime_presence', 'headquarter_country',
                'ss_main_office_state', 'ss_hq_physical_address', 'ngo_vehicle_plate_prefix',
                'year_operations_began', 'website'
            )
        }),
        ('Step 2: Organization Other Information', {
            'fields': (
                'rrc_certificate', 'rrc_registration_number', 'rrc_certificate_start_date',
                'rrc_certificate_expiry_date', 'is_woman_led', 'international_staff_count',
                'national_staff_count', 'relocatable_staff_count', 'juba_residence_compound',
                'annual_operating_budget_usd', 'invoicing_contact_name', 'invoicing_contact_email',
                'invoicing_contact_phone'
            )
        }),
        ('Step 3: Mailing List Contacts', {
            'fields': ('mailing_list_contacts',),
            'classes': ('collapse',)
        }),
        ('Step 4: Code of Conducts', {
            'fields': ('psea_policy', 'codes_of_conduct', 'global_networks'),
            'classes': ('collapse',)
        }),
        ('Step 5: Operational Presence', {
            'fields': ('clusters_sectors', 'operational_presence'),
            'classes': ('collapse',)
        }),
        ('Review & Status', {
            'fields': ('submitted_date', 'reviewed_date', 'reviewer', 'reviewer_notes', 'approved_organization')
        }),
        ('System', {
            'fields': ('created_at', 'updated_at'),
            'classes': ('collapse',)
        }),
    )
    
    actions = ['move_to_pending_payment', 'approve_application', 'reject_application', 'move_to_steering_committee', 'move_to_country_directors']
    
    def move_to_pending_payment(self, request, queryset):
        """Move applications to pending payment status and send payment instructions"""
        from django.utils import timezone
        for application in queryset:
            if application.application_status != 'PENDING_PAYMENT':
                application.application_status = 'PENDING_PAYMENT'
                application.reviewed_date = timezone.now()
                application.reviewer = request.user
                application.save()
                # This will trigger account creation and email via signal or we handle inline
                self._handle_pending_payment(request, application)
        self.message_user(request, f"{queryset.count()} applications moved to Pending Payment. Payment instructions sent.")
    move_to_pending_payment.short_description = "Move to Pending Payment (send payment instructions)"
    
    def approve_application(self, request, queryset):
        from django.utils import timezone
        queryset.update(application_status='APPROVED', reviewed_date=timezone.now(), reviewer=request.user)
        self.message_user(request, f"{queryset.count()} applications approved")
    approve_application.short_description = "Approve selected applications"
    
    def reject_application(self, request, queryset):
        from django.utils import timezone
        queryset.update(application_status='REJECTED', reviewed_date=timezone.now(), reviewer=request.user)
        self.message_user(request, f"{queryset.count()} applications rejected")
    reject_application.short_description = "Reject selected applications"
    
    def move_to_steering_committee(self, request, queryset):
        queryset.update(application_status='STEERING_COMMITTEE')
        self.message_user(request, f"{queryset.count()} applications moved to Steering Committee Review")
    move_to_steering_committee.short_description = "Move to Steering Committee Review"
    
    def move_to_country_directors(self, request, queryset):
        queryset.update(application_status='COUNTRY_DIRECTORS')
        self.message_user(request, f"{queryset.count()} applications moved to Country Directors Vote")
    move_to_country_directors.short_description = "Move to Country Directors Vote"


@admin.register(MembershipPayment)
class MembershipPaymentAdmin(admin.ModelAdmin):
    list_display = ['organization', 'amount', 'payment_date', 'status', 'transaction_reference']
    list_filter = ['status', 'payment_date']
    search_fields = ['organization__name', 'transaction_reference']
    readonly_fields = ['payment_date', 'created_at', 'updated_at']
    date_hierarchy = 'payment_date'


@admin.register(MoMoPaymentRequest)
class MoMoPaymentRequestAdmin(admin.ModelAdmin):
    list_display = ['external_id', 'application', 'amount', 'currency', 'payer_phone', 'status', 'requested_at']
    list_filter = ['status', 'currency', 'requested_at']
    search_fields = ['external_id', 'application__organization_name', 'payer_phone', 'financial_transaction_id']
    readonly_fields = ['external_id', 'reference_id', 'financial_transaction_id', 'requested_at', 'completed_at', 'created_at', 'updated_at']
    date_hierarchy = 'requested_at'
    
    fieldsets = (
        ('Application', {
            'fields': ('application',)
        }),
        ('Payment Details', {
            'fields': ('amount', 'currency', 'payer_phone', 'payer_message', 'payee_note')
        }),
        ('MTN References', {
            'fields': ('external_id', 'reference_id', 'financial_transaction_id')
        }),
        ('Status', {
            'fields': ('status', 'status_reason', 'requested_at', 'completed_at')
        }),
        ('System', {
            'fields': ('created_at', 'updated_at'),
            'classes': ('collapse',)
        }),
    )
    
    actions = ['check_payment_status', 'initiate_payment_request']
    
    def check_payment_status(self, request, queryset):
        """Check payment status from MTN API"""
        from .momo_service import get_momo_service
        
        momo = get_momo_service()
        updated = 0
        
        for payment in queryset.filter(status='PENDING'):
            if payment.reference_id:
                result = momo.get_payment_status(str(payment.reference_id))
                if result['success']:
                    old_status = payment.status
                    payment.status = result['status']
                    if result.get('financial_transaction_id'):
                        payment.financial_transaction_id = result['financial_transaction_id']
                    if result.get('reason'):
                        payment.status_reason = result['reason']
                    
                    if result['status'] == 'SUCCESSFUL':
                        payment.completed_at = timezone.now()
                        # Auto-approve the application
                        self._auto_approve_application(payment.application)
                    
                    payment.save()
                    updated += 1
        
        self.message_user(request, f"Checked {updated} payments. Status updated.")
    check_payment_status.short_description = "Check payment status from MTN"
    
    def initiate_payment_request(self, request, queryset):
        """Initiate a payment request to MTN for pending payments"""
        from .momo_service import get_momo_service
        
        momo = get_momo_service()
        initiated = 0
        
        for payment in queryset.filter(status='PENDING', reference_id__isnull=True):
            result = momo.request_to_pay(
                amount=str(int(payment.amount)),
                currency=payment.currency,
                external_id=payment.external_id,
                payer_phone=payment.payer_phone,
                payer_message=payment.payer_message,
                payee_note=payment.payee_note,
            )
            
            if result['success']:
                import uuid
                payment.reference_id = uuid.UUID(result['reference_id'])
                payment.save()
                initiated += 1
            else:
                messages.warning(request, f"Failed to initiate payment for {payment.external_id}: {result.get('error')}")
        
        self.message_user(request, f"Initiated {initiated} payment requests.")
    initiate_payment_request.short_description = "Send payment request to MTN"
    
    def _auto_approve_application(self, application):
        """Automatically approve application after successful payment"""
        if application.application_status == 'PENDING_PAYMENT':
            application.application_status = 'APPROVED'
            application.save()
            
            # Activate the member organization
            if application.approved_organization:
                application.approved_organization.status = 'ACTIVE'
                application.approved_organization.membership_fee_paid = True
                application.approved_organization.save()
                
                # Send approval confirmation email
                self._send_payment_confirmation_email(application)
    
    def _send_payment_confirmation_email(self, application):
        """Send confirmation email after payment is received"""
        subject = 'South Sudan NGO Forum - Payment Received - Membership Activated!'
        
        message = f"""Dear {application.invoicing_contact_name or application.organization_name},

Congratulations! We have received your membership payment for {application.organization_name}.

Your membership is now ACTIVE!

You can now login to the member portal with full access at:
https://ngo.intvault.online/portal/login

As a member, you now have access to:
- Member directory
- Forums and discussions
- Events and workshops
- Resources and documents
- Job postings
- And much more!

If you have any questions, please contact us at info@southsudanngoforum.org

Welcome to the South Sudan NGO Forum family!

Best regards,
South Sudan NGO Forum Secretariat
"""
        
        try:
            send_mail(
                subject=subject,
                message=message,
                from_email=settings.DEFAULT_FROM_EMAIL,
                recipient_list=[application.invoicing_contact_email],
                fail_silently=False,
            )
        except Exception as e:
            print(f"Payment confirmation email failed: {str(e)}")
